﻿// $Id: IntegerParameterControl.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using VideoLib.Parameters;

namespace RealTimeStereoTestViewer
{
    /// <summary>
    /// A control to modify an integer parameter.
    /// </summary>
    public partial class IntegerParameterControl : UserControl, IParameterControl
    {
        private ParameterInfo paramInfo;
        private object parameter;
        private List<object> followers;

        private int paramRange = 100;
        private int paramMin = 0;

        /// <summary>
        /// Initializes a new instance of the <see cref="ContinuousIntParameterControl"/> class.
        /// </summary>
        /// <param name="parameter">The parameter.</param>
        /// <param name="target">The target.</param>
        public IntegerParameterControl(ParameterInfo paramInfo, object parameter)
            : this(paramInfo, parameter, null)
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="ContinuousIntParameterControl"/> class.
        /// </summary>
        /// <param name="paramInfo">The parameter info.</param>
        /// <param name="parameter">The parameter.</param>
        /// <param name="followers">Any followers.</param>
        public IntegerParameterControl(ParameterInfo paramInfo, object parameter, List<object> followers)
        {
            InitializeComponent();

            // copy arguments
            this.paramInfo = paramInfo;
            this.parameter = parameter;
            this.followers = followers ?? new List<object>(); // replace by empty list if null

            // set minimum and range from parameter info
            if (paramInfo.Maximum != null && paramInfo.Minimum != null)
            {
                paramMin = Convert.ToInt32(paramInfo.Minimum);
                paramRange = Convert.ToInt32(paramInfo.Maximum) - paramMin;
            }

            // set parameter name
            nameLabel.Text = paramInfo.Name;

            // set the trackbar to the current value of the parameter
            int value = paramInfo.GetValue<int>(parameter);
            SetValueLabel(value);

            // set up the track bar
            slider.Minimum = 0;
            slider.Maximum = paramRange;
            int tickpos = (paramRange <= 0 ? 0 : (int)(slider.Maximum * (value - paramMin) / paramRange));
			slider.Value = Math.Min(slider.Maximum, Math.Max(slider.Minimum, tickpos));
            slider.ValueChanged += new EventHandler(slider_ValueChanged);
        }

        /// <summary>
        /// Handles the ValueChanged event of the slider control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void slider_ValueChanged(object sender, EventArgs e)
        {
            if (parameter != null)
            {
                int value = slider.Value + paramMin;
                paramInfo.SetValue<int>(parameter, value);
                SetValueLabel(value);

                foreach (object follower in followers)
                    paramInfo.SetValue<int>(follower, value);
            }
        }

        /// <summary>
        /// Sets the value label.
        /// </summary>
        /// <param name="val">The value.</param>
        private void SetValueLabel(int val)
        {
            valueLabel.Text = val.ToString();
        }

        /// <summary>
        /// Handles the Click event of the resetButton control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void resetButton_Click(object sender, EventArgs e)
        {
            slider.Value = (paramRange <= 0 ? 0 : (int)(slider.Maximum * (Convert.ToInt32(paramInfo.Default) - paramMin) / paramRange));
        }

        #region IParameterControl Members

        public void RefreshValue()
        {
            // set the trackbar to the current value of the parameter
            int value = paramInfo.GetValue<int>(parameter);
            SetValueLabel(value);
            slider.Value = (paramRange <= 0 ? 0 : (int)(slider.Maximum * (value - paramMin) / paramRange));
        }

        #endregion
    }
}
