#pragma once

// Comment out to remove error reporting on CUDA_CALL, and CHECK_KERNEL_ERROR
#define DEBUG
// Comment out to remove logging on RECORD_KERNEL_LAUNCH
//#define DEBUG_INFO

#include <builtin_types.h>
#include "UnmanagedCodeException.h"

// Error handling

// The RECORD_KERNEL_LAUNCH macro is used specifically to log kernel launches
// The LOG_EVENT macro can be used anywhere to write out information
#ifdef DEBUG_INFO
#define RECORD_KERNEL_LAUNCH(name, grid, block) RecordKernelLaunch((name), (grid), (block));
#define LOG_EVENT(description) LogEvent(description);
#else
#define RECORD_KERNEL_LAUNCH(name, grid, block) ;
#define LOG_EVENT(description) ;
#endif//DEBUG_INFO

// The CUDA_CALL macro is used to handle the results from CUDA functions apart from kernel launches
// The CHECK_KERNEL_ERROR macro is used after kernel launches to synchronize and 'get the error in the right place'
#ifdef DEBUG
#define CUDA_CALL(func) CudaCall((func), #func, __FUNCTION__, __FILE__, __LINE__);
#define CHECK_KERNEL_ERROR(name) CheckCudaError(name);
#else
#define CUDA_CALL(func) (func);
#define CHECK_KERNEL_ERROR(name) ;
#endif //DEBUG

// Error handling helper functions (not to be used directly - use with the macros defined above)
bool CudaCall(cudaError error, const char* functionName, const char* surroundingFunction, const char* file, int line);
void RecordKernelLaunch(const char* kernelName, dim3 gridDimension, dim3 blockDimension);
bool CheckCudaError(const char* kernelName);
void LogEvent(const char* description);