﻿// $Id: MiddleburyEvaluationNode.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VideoLib;
using VideoLib.Frames;
using System.Drawing;

namespace CrossChecker
{
	public class MiddleburyEvaluationNode : Node
	{
		public float Threshold { get; set; }
		public string MaskNonoccFilename { get; set; }
		public string MaskAllFilename { get; set; }
		public string MaskDiscFilename { get; set; }

		public MiddleburyEvaluationNode()
			: base()
		{
			Threshold = 1.0f;

			AddInputPins("in_disparities", "in_groundtruth");
			AddOutputPins("out_error_nonocc", "out_error_all", "out_error_disc");
		}

		public override void Process()
		{
			// check input frames
			var frames = GetInputFrames("in_disparities", "in_groundtruth");
			if (PassOnEndOfStreamFrame(frames)) return;
			var bitmapFrames = CastFramesTo<BitmapFrame>(frames);
			AssertFrameProperties(bitmapFrames, FrameProps.Size | FrameProps.Pixelformat | FrameProps.Sequencenumber);
			AssertPixelFormat(bitmapFrames, PixelFormat.Float);

			// load masks
			Bitmap MaskNonocc = (MaskNonoccFilename == null ? null : new Bitmap(MaskNonoccFilename));
			Bitmap MaskAll    = (MaskAllFilename    == null ? null : new Bitmap(MaskAllFilename));
			Bitmap MaskDisc   = (MaskDiscFilename   == null ? null : new Bitmap(MaskDiscFilename));

			float nonocc = 0.0f, all = 0.0f, disc = 0.0f;
			float               everything = EvaluateDisparityMap(bitmapFrames["in_disparities"], bitmapFrames["in_groundtruth"], null,       Threshold);
			if (MaskNonocc != null) nonocc = EvaluateDisparityMap(bitmapFrames["in_disparities"], bitmapFrames["in_groundtruth"], MaskNonocc, Threshold);
			if (MaskAll    != null) all    = EvaluateDisparityMap(bitmapFrames["in_disparities"], bitmapFrames["in_groundtruth"], MaskAll,    Threshold);
			if (MaskDisc   != null) disc   = EvaluateDisparityMap(bitmapFrames["in_disparities"], bitmapFrames["in_groundtruth"], MaskDisc,   Threshold);

			Console.WriteLine("Results: {0:#0.00}%   {1:#0.00}%   {2:#0.00}%   {3:#0.00}% ", nonocc, all, disc, everything);
		}

		private static unsafe float EvaluateDisparityMap(BitmapFrame disparityMap, BitmapFrame groundTruth, Bitmap mask, float threshold)
		{
			int numBadPixels = 0;
			int numTotalPixels = 0;
			int width = disparityMap.Width;
			int height = disparityMap.Height;

			if (mask == null)
			{
				// iterate over every pixel, counting error pixels
				fixed(byte* pbDisp = disparityMap.Data, pbTruth = groundTruth.Data)
				{
					float* pDisp = (float*)pbDisp;
					float* pTruth = (float*)pbTruth;

					for (int y = 0; y < height; y++)
					{
						for (int x = 0; x < width; x++)
						{
							if (Math.Abs(pDisp[width * y + x] - pTruth[width * y + x]) > threshold + 1e-4f)
								numBadPixels++;
							numTotalPixels++;
						}
					}
				}
			}
			else
			{
				// iterate over all masked pixels, counting error pixels
				fixed (byte* pbDisp = disparityMap.Data, pbTruth = groundTruth.Data)
				{
					float* pDisp = (float*)pbDisp;
					float* pTruth = (float*)pbTruth;

					for (int y = 0; y < height; y++)
					{
						for (int x = 0; x < width; x++)
						{
							if (mask.GetPixel(x, y).R == 255) // only consider white pixels (on a black-and-white mask)
							{
								if (Math.Abs(pDisp[width * y + x] - pTruth[width * y + x]) > threshold + 1e-4f)
									numBadPixels++;
								numTotalPixels++;
							}
						}
					}
				}
			}

			return 100.0f * (float)numBadPixels / (float)numTotalPixels;
		}
	}
}
