(* ========================================================================= *)
(* METITARSKI PROVER                                                         *)
(*                                                                           *)
(* Copyright (c) 2007-2011 Lawrence C Paulson                                *)
(* Copyright (c) 2001-2009 Joe Hurd                                          *)
(*                                                                           *)
(* MetiTarski is free software; you can redistribute it and/or modify        *)
(* it under the terms of the GNU General Public License as published by      *)
(* the Free Software Foundation; either version 2 of the License, or         *)
(* (at your option) any later version.                                       *)
(*                                                                           *)
(* MetiTarski is distributed in the hope that it will be useful,             *)
(* but WITHOUT ANY WARRANTY; without even the implied warranty of            *)
(* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *)
(* GNU General Public License for more details.                              *)
(*                                                                           *)
(* You should have received a copy of the GNU General Public License         *)
(* along with MetiTarski; if not, write to the Free Software                 *)
(* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *)
(* ========================================================================= *)

open Useful;

(* ------------------------------------------------------------------------- *)
(* The program name.                                                         *)
(* ------------------------------------------------------------------------- *)

val PROGRAM = "metit";

(* ------------------------------------------------------------------------- *)
(* Program options.                                                          *)
(* ------------------------------------------------------------------------- *)

val QUIET = ref false;

val TEST = ref false;

val TPTP : string option ref = ref NONE;

val cpu_limit = ref 600;

val autoInclude = ref false;
val extended = ref false;
val superExtended = ref false;

val ITEMS = ["name","goal","clauses","size","category","proof","saturation"];

val extended_items = "all" :: ITEMS;

val show_items = map (fn s => (s, ref false)) ITEMS;

fun show_ref s =
    case List.find (equal s o fst) show_items of
      NONE => raise Bug ("item " ^ s ^ " not found")
    | SOME (_,r) => r;

fun show_set b = app (fn (_,r) => r := b) show_items;

fun showing s = not (!QUIET) andalso (s = "status" orelse !(show_ref s));

fun notshowing s = not (showing s);

fun showing_any () = List.exists showing ITEMS;

fun notshowing_any () = not (showing_any ());

fun show "all" = show_set true
  | show s = case show_ref s of r => r := true;

fun hide "all" = show_set false
  | hide s = case show_ref s of r => r := false;

(* ------------------------------------------------------------------------- *)
(* Process command line arguments and environment variables.                 *)
(* ------------------------------------------------------------------------- *)
(* Default values - these are not refs *)
val max_splits_r = ~1
val atom_weight_r = 0;
val maxStackDepth =  100;
val SOSweightFactor = 2.0;

(* selects what type of case splitting to do *)
val backTracking = ref true
val caseSplitting = ref true
(* if cases parameters have been actively set then this will *)
(* over-ride the defaults set above.                         *)
val casesParam1Set = ref false
val casesParam1 = ref 0
val casesParam2Set = ref false
val casesParam2 = ref 0

fun parse_cases_params s =
  if s = "off" then
     let
         val _ = caseSplitting := false
     in
         (0,10)
     end
  else
     case map Int.fromString (String.tokens (fn c => c = #"+") s) of
       [SOME ms]          => let val _ = casesParam1Set := true in (ms, 0) end
     | [SOME ms, SOME aw] => let val _ = casesParam1Set := true val _ = casesParam2Set := true in (ms, aw) end
     | _ => die "case split parameters have the form INT+INT, INT, or \"off\""

fun set_cases_params s =
  let
     val (ms, aw) = parse_cases_params s
  in  casesParam1 := ms; casesParam2 := aw  end;

local
  open Useful Options;
in
  val specialOptions =
    [{switches = ["--show"], arguments = ["ITEM"],
      description = "show ITEM (see below for list)",
      processor =
        beginOpt (enumOpt extended_items endOpt) (fn _ => fn s => show s)},


     {switches = ["--autoInclude"], arguments = [],
      description = "Automatically select axiom files to include",
      processor = beginOpt endOpt (fn _ => (autoInclude := true; extended := false; superExtended := false))},

     {switches = ["--autoIncludeExtended"], arguments = [],
      description = "auto includes extended axioms",
      processor = beginOpt endOpt (fn _ => (autoInclude := true; extended := true; superExtended := false))},

     {switches = ["--autoIncludeSuperExtended"], arguments = [],
      description = "auto includes super extended axioms",
      processor = beginOpt endOpt (fn _ => (autoInclude := true;extended := false; superExtended := true))},



     {switches = ["--hide"], arguments = ["ITEM"],
      description = "hide ITEM (see below for list)",
      processor =
        beginOpt (enumOpt extended_items endOpt) (fn _ => fn s => hide s)},
     {switches = ["-p"], arguments = [],
      description = "show proof",
      processor = beginOpt endOpt (fn _ => show "proof")},
     {switches = ["--time"], arguments = ["positive integer"],
      description = "processor time limit (in seconds)",
      processor = intOpt (SOME 0, NONE) endOpt (fn k => cpu_limit := k)},
     {switches = ["--maxweight","-w"], arguments = ["positive integer"],
      description = "maximum weight of a retained clause",
      processor = intOpt (SOME 0, NONE) endOpt Waiting.set_maxweight},
     {switches = ["--maxalg"], arguments = ["positive integer"],
      description = "maximum number of symbols in an algebraic clause",
      processor = intOpt (SOME 0, NONE) endOpt Resolution.set_maxalg},
(* ------------------------------------------------------------------------ *)
(* maximum run of nonSOS clauses allowed before giving up ...               *)
     {switches = ["--maxnonSOS"], arguments = ["positive integer"],
      description = "maximum run of non SOS given clauses allowed before giving up",
      processor = intOpt (SOME 0, NONE) endOpt Resolution.setMaxRunOutsideSOS},
(* ------------------------------------------------------------------------ *)
(*  Turns on re-running with high value of maxalg if waiting becomes empty  *)
     {switches = ["--rerun"], arguments = ["off/ON"],
      description = "before giving up, rerun with high maxalg (default is on)",
      processor = beginOpt (enumOpt ["off","on"] endOpt)
                    (fn _ => fn s => Resolution.rerunWithHighMaxAlg (s="on") )},
(* ------------------------------------------------------------------------ *)
     {switches = ["--tptp"], arguments = ["DIR"],
      description = "specify the TPTP installation directory",
      processor =
        beginOpt (stringOpt endOpt) (fn _ => fn s => TPTP := SOME s)},
     {switches = ["--tstp"], arguments = [],
      description = "generate standard TSTP: no infixes, etc.",
      processor = beginOpt endOpt (fn _ => Tptp.standardTSTP := true)},
(* ------------------- Paramodulation --------------------- *)
     {switches = ["--paramodulation"], arguments = ["off/ON"],
      description = "turn full paramodulation off (default is on)",
      processor =
        beginOpt (enumOpt ["off","on"] endOpt) (fn _ => fn s => Clause.useParamodulation := (s="on"))},
(* -------------------------------------------------------- *)
     {switches = ["--cases"], arguments = ["#cases+weight"],
      description = "max permitted active case splits/nonSOS weighting factor in tenths (10 = neutral weighting)",
      processor =
        beginOpt (stringOpt endOpt) (fn _ => fn s => set_cases_params s)},
(* ---------------- This is case splitting with backtracking -------------- *)
     {switches = ["--backtracking"], arguments = ["off/ON"],
      description = "turn backtracking off (default is on)",
      processor =
        beginOpt (enumOpt ["off","on"] endOpt) (fn _ => fn s => backTracking := (s="on"))},
(* --------------- This enables projection ordering for CAD --------------- *)
     {switches = ["--proj_ord"], arguments = ["off/ON"],
      description = "switch CAD projection ordering off (default is on)",
      processor = beginOpt (enumOpt ["off","on"] endOpt)
                    (fn _ => fn s => RCF.set_proj_order (s="on"))},
(* --------------- This enables polynomial ICP before EADM -------------- *)
     {switches = ["--icp_eadm"], arguments = [],
      description = "enable polynomial ICP before EADM",
      processor = beginOpt endOpt (fn _  => RCF.set_adm_coop_method(RCF.ICP_THEN_EADM))},
(* -------------- This enables only polynomial ICP, no EADM ------------- *)
     {switches = ["--icp"], arguments = [],
      description = "enable only polynomial ICP, no EADM",
      processor = beginOpt endOpt (fn _  => RCF.set_adm_coop_method(RCF.ICP_ONLY))},
(* ------- This enables Mathematica as the EADM -------- *)
     {switches = ["-m", "--mathematica"], arguments = [],
      description = "use Mathematica as EADM",
      processor = beginOpt endOpt (fn _  => RCF.set_eadm(RCF.MATHEMATICA))},
(* ------- This enables SMT solver (Z3_nonlin) as the EADM -------- *)
     {switches = ["--smt"], arguments = [],
      description = "use SMT solver (Z3_nonlin) as EADM",
      processor = beginOpt endOpt (fn _  => RCF.set_eadm(RCF.SMT_Z3))},
(* -------------- This enables ICP-based SAT search for RCF formulas------- *)
     {switches = ["--icp_sat"], arguments = [],
      description = "use ICP to search for RCF counter-example before refutation",
      processor = beginOpt endOpt (fn _  => RCF.check_icp_sat := true)},
(* ----- This enables univ. relaxation for SAT filtration on RCF formulas ----- *)
     {switches = ["--univ_sat"], arguments = [],
      description = "use univariate relaxations for RCF SAT checks (EADM only)",
      processor = beginOpt endOpt (fn _  => RCF.check_univ_rcf_sat := true)},
(* ----- Set RCF strategy ID ----- *)
     {switches = ["--strategy"], arguments = ["positive integer"],
      description = "ID of RCF decision strategy",
      processor = intOpt (SOME 0, NONE) endOpt (fn k => RCF.set_strategy_id k)},
     {switches = ["--unsafe_divisors"], arguments = [],
      description = "don't verify that divisors are nonzero",
      processor = beginOpt endOpt (fn _ => Poly.unsafe_divisors := true)},
     {switches = ["--full"], arguments = [],
      description = "include variable instantiations in proofs",
      processor = beginOpt endOpt (fn _ => Tptp.fullProofs := true)},
     {switches = ["-q","--quiet"], arguments = [],
      description = "Run quietly; indicate provability with return value",
      processor = beginOpt endOpt (fn _ => (traceLevel := 0; QUIET := true))},
     {switches = ["--test"], arguments = [],
      description = "Skip the proof search for the input problems",
      processor = beginOpt endOpt (fn _ => TEST := true)},
     {switches = ["--ml_calibrate"], arguments = [],
      description = "Calibrate EADM initialisation CPU time for machine learning",
      processor = beginOpt endOpt (fn _ => (Calibrate.calibrate_eadms 50; TEST := true))},
     {switches = ["--ml_trace"], arguments = [],
      description = "Perform EADM strategy tracing for machine learning",
      processor = beginOpt endOpt (fn _ => (RCF.trace_machine_learning := true))}];
end;

(*LCP:  version numbering and date stamp*)
val dateString = Date.fmt "%d %b %Y" (Date.fromTimeLocal (Time.now()));

val versionString = "MetiTarski 2.0 (built "^ dateString ^")"^"\n";

val programOptions =
    {name = PROGRAM,
     version = versionString,
     header = "usage: "^PROGRAM^" [option ...] problem.tptp ...\n" ^
              "Proves the input TPTP problem files.\n",
     footer = "Possible ITEMs are {" ^ join "," extended_items ^ "}.\n" ^
              "Problems can be read from standard input using the " ^
              "special - filename.\n",
     options = specialOptions @ Options.basicOptions};

fun exit x : unit = Options.exit programOptions x;
fun succeed () = Options.succeed programOptions;
fun fail mesg = Options.fail programOptions mesg;
fun usage mesg = Options.usage programOptions mesg;

fun processOptions () =
    let
      val args = CommandLine.arguments ()

      val (_,work) = Options.processOptions programOptions args

      val () =
          case !TPTP of
            SOME _ => ()
          | NONE => TPTP := OS.Process.getEnv "TPTP"
    in
      work
    end;

(* ------------------------------------------------------------------------- *)
(* Top level. LCP:  reporting runtime and killing child processes            *)
(* ------------------------------------------------------------------------- *)

exception Timeout;

(*The QEPCAD process must be closed prior to calling this function,
  otherwise its runtime will appear to be zero.*)
fun runtime_msg () =
  let val {utime,cutime, ...} = Posix.ProcEnv.times()
      val rcf_time = if Time.>(!RCF.extra_cpu_time, Time.zeroTime) then 
			 case !RCF.strategy_id of
			     NONE => !RCF.extra_cpu_time (* Mathematica time *)
			   | _ => cutime (* Everything except Mathematica *)
		     else cutime
      val tot_utime = Time.+(utime, rcf_time)
  in  "Processor time: " ^
      Time.toString tot_utime ^ " = " ^
      Time.toString utime ^ " (Metis) + " ^ (Time.toString rcf_time) ^ " (RCF)\n"
  end;
    
(*Prevents the printing of multiple exit messages, caused by exceptions raised
  by the killing of the subprocess. Oddly, exception handlers don't catch these.*)
val exitr = ref false;

(*Close, print runtime, quit.*)
fun fail_cleanly msg =
 (if !exitr then () else (exitr := true; TextIO.closeOut TextIO.stdErr; print msg);
  RCF.kill_in_progress := true;
  RCF.close_with_time true; print (runtime_msg());
  OS.Process.exit OS.Process.failure);

fun string_of_exn (OS.SysErr (msg,_)) = "Operating System error: " ^ msg
  | string_of_exn e = exnMessage e;

(* ------------------------------------------------------------------------- *)
(* The core application.                                                     *)
(* ------------------------------------------------------------------------- *)

(*MetisDebug
val next_cnf =
    let
      val cnf_counter = ref 0
    in
      fn () =>
         let
           val ref cnf_count = cnf_counter
           val () = cnf_counter := cnf_count + 1
         in
           cnf_count
         end
    end;
*)

local
  fun display_sep () =
      if notshowing_any () then ()
      else print (nChars #"-" (!Print.lineLength) ^ "\n");

  fun display_name filename =
      if notshowing "name" then ()
      else print ("Problem: " ^ filename ^ "\n\n");

  fun display_goal tptp =
      if notshowing "goal" then ()
      else
        let
          val goal = Tptp.goal tptp
        in
          print ("Goal:\n" ^ Formula.toString goal ^ "\n\n")
        end;

  fun display_clauses cls =
      if notshowing "clauses" then ()
      else print ("Clauses:\n" ^ Problem.toString cls ^ "\n\n");

  fun display_size cls =
      if notshowing "size" then ()
      else
        let
          fun plural 1 s = "1 " ^ s
            | plural n s = Int.toString n ^ " " ^ s ^ "s"

          val {clauses,literals,symbols,typedSymbols} = Problem.size cls
        in
          print
            ("Size: " ^
             plural clauses "clause" ^ ", " ^
             plural literals "literal" ^ ", " ^
             plural symbols "symbol" ^ ", " ^
             plural typedSymbols "typed symbol" ^ ".\n\n")
        end;

  fun display_category cls =
      if notshowing "category" then ()
      else
        let
          val cat = Problem.categorize cls
        in
          print ("Category: " ^ Problem.categoryToString cat ^ ".\n\n")
        end;

  local
    fun display_proof_start filename =
        print ("\nSZS output start CNFRefutation for " ^ filename ^ "\n");

    fun display_proof_body problem proofs =
        let
          val comments = []

          val includes = []

          val formulas =
              Tptp.fromProof
                {problem = problem,
                 proofs = proofs}

          val proof =
              Tptp.Problem
                {comments = comments,
                 includes = includes,
                 formulas = formulas}

          val mapping = Tptp.defaultMapping
          val mapping = Tptp.addVarSetMapping mapping (Tptp.freeVars proof)

          val filename = "-"
        in
          Tptp.write
            {problem = proof,
             mapping = mapping,
             filename = filename}
        end;

    fun display_proof_end filename =
        print ("SZS output end CNFRefutation for " ^ filename ^ "\n\n");
  in
    fun display_proof filename problem proofs =
        if notshowing "proof" then ()
        else
          let
            val () = display_proof_start filename
            val () = display_proof_body problem proofs
            val () = display_proof_end filename
          in
            ()
          end;
  end;

  fun display_saturation filename ths =
      if notshowing "saturation" then ()
      else
        let
(*MetisDebug
          val () =
              let
                val problem =
                    Tptp.mkProblem
                      {comments = ["Saturation clause set for " ^ filename],
                       includes = [],
                       names = Tptp.noClauseNames,
                       roles = Tptp.noClauseRoles,
                       problem = {axioms = [],
                                  conjecture = map Thm.clause ths}}

                val mapping =
                    Tptp.addVarSetMapping Tptp.defaultMapping
                      (Tptp.freeVars problem)
              in
                Tptp.write
                  {problem = problem,
                   mapping = mapping,
                   filename = "saturation.tptp"}
              end
*)
          val () = print ("\nSZS output start Saturation for " ^ filename ^ "\n")
          val () = app (fn th => print (Thm.toString th ^ "\n")) ths
          val () = print ("SZS output end Saturation for " ^ filename ^ "\n\n")
        in
          ()
        end;

  fun display_status filename status =
      if notshowing "status" orelse !exitr then ()
      else print ("SZS status " ^ Tptp.toStringStatus status ^
                  " for " ^ filename ^ "\n");

  fun display_problem filename cls =
      let
(*MetisDebug
        val () =
            let
              val problem =
                  Tptp.mkProblem
                    {comments = ["CNF clauses for " ^ filename],
                     includes = [],
                     names = Tptp.noClauseNames,
                     roles = Tptp.noClauseRoles,
                     problem = cls}

              val mapping =
                  Tptp.addVarSetMapping Tptp.defaultMapping
                    (Tptp.freeVars problem)

              val filename = "cnf_" ^ Int.toString (next_cnf ()) ^ ".tptp"
            in
              Tptp.write
                {problem = problem,
                 mapping = mapping,
                 filename = filename}
            end
*)
        val () = display_clauses cls
        val () = display_size cls
        val () = display_category cls
      in
        ()
      end;

  fun mkTptpFilename filename =
    let val filename = stripSuffix (equal #" ") (stripPrefix (equal #" ") filename)
    in
      case !TPTP of
        NONE => filename
      | SOME tptp =>
      	  if String.isPrefix "/" filename then filename (*Absolute path name*)
      	  else
	    let val tptp = stripSuffix (equal #"/") tptp
	    in
	      tptp ^ "/" ^ filename
	    end
    end;

  fun readIncludes mapping seen formulas includes =
      case includes of
        [] => formulas
      | inc :: includes =>
        if StringSet.member inc seen then
          readIncludes mapping seen formulas includes
        else
          let
            val seen = StringSet.add seen inc

            val filename = mkTptpFilename inc

            val Tptp.Problem {includes = i, formulas = f, ...} =
                MT_Tptp.read {filename = filename, mapping = mapping}

            val formulas = f @ formulas

            val includes = List.revAppend (i,includes)
          in
            readIncludes mapping seen formulas includes
          end;

(* -------------------------------------------------------------------- *)
(* Utility functions for code to automatically determine axiom includes *)
(* -------------------------------------------------------------------- *)

  fun writeOutIncludes [] = print "\n"
    | writeOutIncludes (inc::includes) = 
          let
             val _ = print(inc ^ "\n")
          in
             writeOutIncludes includes
          end;

  fun getThmList' [] acc = acc
    | getThmList' ({problem={axioms,conjecture},...}::rest) acc =
           let
               val axioms = map Thm.axiom axioms
               val conjecture = map Thm.conjecture conjecture
               val acc = acc @ axioms
               val acc = acc @ conjecture
           in
               getThmList' rest acc
           end

   fun getThmList probs = getThmList' probs []

  fun printThmList' [] output = print (output ^ "\n")
    | printThmList' (th::ths) output =
           let
               val output  = output ^ Print.toString Thm.pp th ^ "\n"
           in
               printThmList' ths output
           end;
  fun printThmList thmList = printThmList' thmList "\nConjecture Clause List\n";

  fun getFunctionSet [] fx = fx
    | getFunctionSet (th::ths) fx =
           let
                fun addLits [] fx = fx
                 |  addLits (lit::lits) fx =
                       let
                            fun onlyArity1plus nameArity =
                                (NameArity.arity nameArity > 0) orelse
                                (NameArity.arity nameArity = 0 andalso 
                                 NameArity.name nameArity= "pi")
                                
                            val name_arity_set = Literal.functions lit
                            val name_arity_set = NameAritySet.filter onlyArity1plus name_arity_set
                            val fx = NameAritySet.union name_arity_set fx
                       in
                            addLits lits fx
                       end
                val literals = LiteralSet.toList (Thm.clause th)
                val fx = addLits literals fx
            in
                getFunctionSet ths fx
            end;
  fun getFunctionNames thmList =
      let
         val fx = getFunctionSet thmList (NameAritySet.empty)
      in
         NameAritySet.toList fx
      end;

  fun printFunctionNames thmList =
      let
         fun printNameList [] output = print (output ^ "\n")
          |  printNameList (name::names) output =
                  let
                       val output = output ^ (NameArity.name name) ^ "\n"
                  in
                       printNameList names output
                  end
          fun printAll names = printNameList names "List of function names\n"
          val nameList = getFunctionNames thmList
      in
          printAll nameList
      end;

(* Long and rather messy function to actually add the includes *)
   fun addIncludes fx =
       let
          val incs = ["Axioms/general.ax","Axioms/minmax.ax"]
          val incs = if (NameAritySet.member ("abs",1) fx) then
                  incs @
                       ["Axioms/abs.ax"]
                  else incs
          val incs = if (NameAritySet.member ("arccos",1) fx) then
                         if(!extended orelse !superExtended)then
                            incs @ ["Axioms/acos.ax","Axioms/asin-lower.ax","Axioms/asin-upper.ax",
                                    "Axioms/pi.ax","Axioms/sqrt-extended.ax","Axioms/sqrt-general.ax"]
                         else
                            incs @ ["Axioms/acos.ax","Axioms/asin-lower.ax","Axioms/asin-upper.ax",
                                  "Axioms/pi.ax","Axioms/sqrt-general.ax","Axioms/sqrt-lower.ax","Axioms/sqrt-upper.ax"]
                  else incs
          val incs = if (NameAritySet.member ("arcsin",1) fx) then
                          if(!extended orelse !superExtended)then
                            incs @ ["Axioms/asin-lower.ax","Axioms/asin-upper.ax",
                                    "Axioms/sqrt-extended.ax","Axioms/sqrt-general.ax"]
                         else
                            incs @ ["Axioms/asin-lower.ax","Axioms/asin-upper.ax",
                                 "Axioms/sqrt-general.ax","Axioms/sqrt-lower.ax","Axioms/sqrt-upper.ax"]
                  else incs
          val incs = if (NameAritySet.member ("arctan",1) fx) then
                         if(!extended) then
                             incs @["Axioms/atan-extended.ax"]
                         else if (!superExtended) then
                             incs @["Axioms/atan-extended2.ax"]
                         else
                             incs @["Axioms/atan-lower.ax","Axioms/atan-upper.ax"]
                  else incs
          val incs = if (NameAritySet.member ("cbrt",1) fx) then
                  incs @
                       ["Axioms/cbrt-lower.ax","Axioms/cbrt-upper.ax"]
                  else incs
          val incs = if (NameAritySet.member ("cos",1) fx) then
                        if(!extended)then
                            incs @ ["Axioms/cos-extended.ax"]
                        else if(!superExtended) then
                            incs @ ["Axioms/cos-extended2.ax"]
                         else
                            incs @ ["Axioms/cos.ax"]
                   else incs
          val incs = if (NameAritySet.member ("cosh",1) fx) then
                   incs @
                       ["Axioms/cosh-lower.ax","Axioms/cosh-upper.ax"]
                   else incs
          val incs = if (NameAritySet.member ("exp",1) fx) then
                         if(!extended orelse !superExtended)then
                             incs @ ["Axioms/exp-extended.ax","Axioms/exp-general.ax"]
                         else
                             incs @ ["Axioms/exp-general.ax","Axioms/exp-lower.ax","Axioms/exp-upper.ax"]
                   else incs
          val incs = if (NameAritySet.member ("ln",1) fx) then
                         if(!extended orelse !superExtended)then
                             incs @ ["Axioms/ln-extended.ax","Axioms/ln-general.ax"]
                         else
                             incs @ ["Axioms/ln-general.ax","Axioms/ln-lower.ax","Axioms/ln-upper.ax"]
                   else incs
          val incs = if (NameAritySet.member ("pi",0) fx) then
                   incs @
                       ["Axioms/pi.ax"]
                   else incs
          val incs = if (NameAritySet.member ("pow",2) fx) then
                   incs @
                       ["Axioms/pow.ax"]
                   else incs
          val incs = if (NameAritySet.member ("sin",1) fx) then
                         if(!extended)then
                               incs @ ["Axioms/sin-extended.ax"]
                         else if(!superExtended)then
                               incs @ ["Axioms/sin-extended2.ax"]
                         else
                               incs @ ["Axioms/sin.ax"]
                   else incs
          val incs = if (NameAritySet.member ("sinh",1) fx) then
                   incs @
                       ["Axioms/sinh-lower.ax","Axioms/sinh-upper.ax"]
                   else incs
          val incs = if (NameAritySet.member ("sqrt",1) fx) then
                         if(!extended orelse !superExtended)then
                            incs @ ["Axioms/sqrt-extended.ax","Axioms/sqrt-general.ax"]
                         else
                            incs @ ["Axioms/sqrt-general.ax","Axioms/sqrt-lower.ax","Axioms/sqrt-upper.ax"]
                   else incs
          val incs = if (NameAritySet.member ("tan",1) fx) then
                   incs @
                       ["Axioms/tan-lower.ax","Axioms/tan-upper.ax",
                        "Axioms/tan.ax"]
                   else incs
          val incs = if (NameAritySet.member ("tanh",1) fx) then
                   incs @
                       ["Axioms/tanh-lower.ax","Axioms/tanh-upper.ax"]
                   else incs
       in
          incs
       end;

(* ------------------------------------------- *)
(* End of utility functions for axiom includes *)
(* ------------------------------------------- *)


  fun read mapping filename =
      let
        val problem = MT_Tptp.read {filename = filename, mapping = mapping}

        val Tptp.Problem {comments,includes,formulas} = problem
	
        (* ------------------------------------------------------------- *)
	(* code to generate includes automatically from the forumalas    *)
        (* ignore formulas in existing includes - doing this is a design *)
        (* decision - it means that automatic axiom selection can only   *)
        (* apply to conjectures in the main file.                        *)
        (* ------------------------------------------------------------- *)
        val problemMinusIncludes = Tptp.Problem
                                   {comments = comments,
                                    includes = [],formulas = formulas}
        val probs = Tptp.normalize problemMinusIncludes
        val thmList = getThmList probs
        (* val _ = printThmList thmList
        val _ = printFunctionNames thmList *)
        val fx = getFunctionSet thmList (NameAritySet.empty)
        val generatedIncludes = addIncludes fx
        (*
        val _ = print("Generated Include List\n")
        val _ = writeOutIncludes generatedIncludes
        val _ = print("\n")
        *)
        (* ---------------------------------------------------------------------- *)
        (* If autoInclude is set then add the generated includes to the existing  *)
        (* list of includes. This may lead to duplication but this doesn't matter *)
        (* as duplicates are removed in function readIncludes.                    *)
        (* ---------------------------------------------------------------------- *)
        val includes = if(!autoInclude) then (generatedIncludes @ includes) else includes
        (* ---------------------------------------------- *)
	(* end of code to automatically generate includes *)
        (* ---------------------------------------------- *)
      in
        if null includes then problem
        else
          let
            val seen = StringSet.empty

            val includes = rev includes

            (* val _ = writeOutIncludes includes *)

            val formulas = readIncludes mapping seen formulas includes
          in
            Tptp.Problem
              {comments = comments,
               includes = [],
               formulas = formulas}
          end
      end;

  (*LCP: No resolutionParameters, no models....*)

  fun refute {axioms,conjecture} =
      let
        val axioms = map Thm.axiom axioms
        and conjecture = map Thm.conjecture conjecture
        val problem = {axioms = axioms, conjecture = conjecture}
        val resolution = Resolution.new Resolution.default problem
      in
        Resolution.loop resolution
      end;

  fun refuteAll filename tptp probs acc =
      case probs of
        [] =>
        let
          val status =
              if !TEST then Tptp.UnknownStatus
              else if Tptp.hasFofConjecture tptp then Tptp.TheoremStatus
              else Tptp.UnsatisfiableStatus

          val () = display_status filename status

          val () =
              if !TEST then ()
              else display_proof filename tptp (rev acc)
        in
          true
        end
      | prob :: probs =>
        let
          val {subgoal,problem,sources} = prob

          val () = display_problem filename problem
        in
          if !TEST then refuteAll filename tptp probs acc
          else
            case refute problem of
              Resolution.Contradiction th =>
              let
                val subgoalProof =
                    {subgoal = subgoal,
                     sources = sources,
                     refutation = th}

                val acc = subgoalProof :: acc
              in
                refuteAll filename tptp probs acc
              end
            | Resolution.Satisfiable ths =>
              let
                val status = Tptp.GaveUpStatus (*Incomplete, so cannot report "satisfiable"*)
                val () = display_status filename status
                val () = display_saturation filename ths
              in
                false
              end
        end;

  val filename_p = ref "???";
in
  fun prove mapping filename =
      let
	(*For the Poly/ML signal structure, not the SML/NJ one. It will run AS A SEPARATE THREAD*)
	fun sig_xcpu_handler _ =
	      (display_status filename Tptp.TimeoutStatus;
    	       fail_cleanly "Processor time limit exceeded.\n");
	fun sig_int_handler _  = fail_cleanly "Interrupt signal received.\n";
	fun sig_alrm_handler _ = fail_cleanly "Alarm signal received.\n";
        val () = display_sep ()
        val () = display_name filename
        val tptp = read mapping filename
        val () = display_goal tptp
        val problems = Tptp.normalize tptp
	val _ = Signal.signal (RCF.sigxcpu, Signal.SIG_HANDLE sig_xcpu_handler);
	val _ = Signal.signal (Posix.Signal.int, Signal.SIG_HANDLE sig_int_handler);
	val _ = Signal.signal (Posix.Signal.alrm, Signal.SIG_HANDLE sig_alrm_handler);
      in
        filename_p := filename; refuteAll filename tptp problems []
      end
    handle
       IO.Io {name,function,cause} =>
	 (display_status filename Tptp.ErrorStatus;
	  if function = "TextIO.openIn" then (print ("File " ^ name ^ " not found!\n"); false)
	  else fail_cleanly ("Exception IO raised for file " ^ filename ^ "\n" ^
	                "  name = " ^ name ^  ", function = " ^ function ^
		        ", cause = " ^ string_of_exn cause ^ "\n"))
     | Parse.NoParse =>
         (display_status filename Tptp.SyntaxErrorStatus;
	  fail_cleanly ("Syntax error in file " ^ filename ^ "\n"))
     | Timeout =>
         (display_status filename Tptp.TimeoutStatus;
	  fail_cleanly "Processor time limit exceeded.\n")
     | Thread.Thread.Interrupt =>
         (display_status filename Tptp.TimeoutStatus;
	  fail_cleanly "Interrupt signal received.\n")
     | Error s =>
         (display_status filename Tptp.ErrorStatus;
	  fail_cleanly (PROGRAM^" failed: " ^ s ^ ", file = " ^ filename ^ "\n"))
     | Bug s =>
         (display_status filename Tptp.ErrorStatus;
          fail_cleanly ("BUG found in "^PROGRAM^" program:\n" ^ s^ "  file = " ^ filename ^ "\n"))
     | e => (display_status filename Tptp.ErrorStatus;
             fail_cleanly ("Exception " ^ exnName e ^ " raised by "^PROGRAM^" program.\n" ^ "  file = " ^ filename ^ "\n  " ^ string_of_exn e ^ "\n"))

  fun proveAll mapping filenames =
      List.all
        (if !QUIET then prove mapping
         else fn filename => prove mapping filename orelse true)
        filenames;

  (*Creates a new thread that monitors the cpu usage quits within a second of detecting that
    the total usage has exceeded the limit. Also checks wall clock time (with twice the limit)
    because sometimes QEPCAD gets stuck. Thanks to DCJM for help.*)
  fun monitor_cpu secs =
    let val t = Time.fromSeconds secs and tw = Time.fromSeconds (2*secs)
	fun poll (cpuT,realT) =
	  let val {usr, sys} = Timer.checkCPUTimer cpuT
	      and wall       = Timer.checkRealTimer realT
	  in
	    if wall > tw orelse usr > t   (*or, usr+sys to include system kernel time*)
	    then (display_status (!filename_p) Tptp.TimeoutStatus;
		  fail_cleanly "Processor time limit exceeded.\n")
	    else sleeping (cpuT, realT, Time.fromSeconds 1)
	  end
	and sleeping (cpuT,realT,w) = (OS.Process.sleep w; poll (cpuT,realT))
    in
       (*The initial sleep is for the full t seconds, further ones are one second part.*)
       Thread.Thread.fork(fn () => sleeping(Timer.startCPUTimer(), Timer.startRealTimer(), t), [])
    end;

end;

(* ------------------------------------------------------------------------- *)
(* Top level. LCP                                                            *)
(* ------------------------------------------------------------------------- *)

(*LCP: export from Poly/ML requires these declarations to be made within a function!
   also our version performs process communication.*)
fun metis () =
let
  (*BasicDebug val () = print "Running in basic DEBUG mode.\n" *)
  (*MetisDebug val () = print "Running in metis DEBUG mode.\n" *)

  val work = processOptions ()

  (* Set the various case splitting options *)
  val _ = if !caseSplitting andalso !backTracking then
          let
             val _ = if !casesParam1Set then
                     Resolution.maxStackDepth := !casesParam1
                     else
                     Resolution.maxStackDepth := maxStackDepth
             val _ = if !casesParam2Set then
                     Waiting.SOSweightFactor := 0.1 * Real.fromInt (!casesParam2)
                     else
                     Waiting.SOSweightFactor := SOSweightFactor
             val _ = Active.max_splits_r := 0
          in
             ()
          end
          else if !caseSplitting then
          let
             val _ = if !casesParam1Set then
                     Active.max_splits_r := !casesParam1
                     else
                     Active.max_splits_r := max_splits_r
             val _ = if !casesParam2Set then
                     Waiting.SOSweightFactor := 0.1 * Real.fromInt (!casesParam2)
                     else
                     Waiting.SOSweightFactor := 1.0
             val _ = Resolution.maxStackDepth := 0
          in
             ()
          end
          else
          let
              val _ = Active.max_splits_r := 0
              val _ = Resolution.maxStackDepth := 0
              val _ = Waiting.SOSweightFactor := 1.0
          in
             ()
          end
  val _ = Literal.atom_weight_r := atom_weight_r
  (* -------------------------------------------------------------------------- *)
  (* if the maximum weight has not been specifically set then adjust it upwards *)
  (* to allow for the increase from SOSweightFactor.                            *)
  (* -------------------------------------------------------------------------- *)
  val w = !Waiting.maxweight
  val w = if !Waiting.maxweightSet then w else w * (!Waiting.SOSweightFactor)
  val _ = Waiting.maxweight := w

  val _ = if null work then usage "no input problem files"
          else monitor_cpu (!cpu_limit);

  val success = proveAll Tptp.defaultMapping work
in
  RCF.close_with_time false;
  print (runtime_msg());
  if RCF.icp_enabled() then
      print ("# RCF+ formulas refuted by:  EADM = " ^ Int.toString(!RCF.num_refuted_by_eadm)
	     ^ ", ICP = " ^ Int.toString(!RCF.num_refuted_by_icp) ^ ".\n")
  else ();
  print ("Maximum weight in proof search: " ^ Int.toString (Real.floor (!Waiting.maxweight_used)) ^ "\n");
  exit {message = NONE, usage = false, success = success}
end
handle e => fail_cleanly ("Exception " ^ exnName e ^ " raised: " ^ string_of_exn e ^ "\n")


