### TempCov: Toolkit for investigating temperature covert channels
### Copyright (C) 2006 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
###
### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; either version 2 of the License, or
### (at your option) any later version.
###
### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.
###
### You should have received a copy of the GNU General Public License along
### with this program; if not, write to the Free Software Foundation, Inc.,
### 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
###
### $Id: util.R 3381 2006-12-25 18:07:55Z sjm217 $

## Convert cm to inches
inFromCm <- function(cm) {
  return(cm/2.54)
}

## Create a mask which removes points which are very close to each other
## This prevents the vector image generated having a stupidly large filesize
##
## x, y: Vectors to be de-dupeds
## xLimit, yLimit: Maximum number unique values on x, y axes
## xRange, yRange: Range of x and y vectors. If not specified this is calculated
mRemoveDupes <- function(x, y, xLimit=300, yLimit=100, xRange=NULL, yRange=NULL) {
  ## Treat vectors as numeric (even if they are POSIXct)
  class(x) <- "numeric"
  class(y) <- "numeric"

  ## If ranges are not specified, calculate them
  if (is.null(xRange))
    xRange <- range(x)

  if (is.null(yRange))
    yRange <- range(y)

  ## Calculate the length of the range
  xDiff <- diff(xRange)
  yDiff <- diff(yRange)

  ## Scale all elements so length of range is 1
  x <- x / xDiff
  y <- y / yDiff

  ## Scale all elements so length of range is xLimit/yLimit
  x <- x * xLimit
  y <- y * yLimit

  ## Round all elements so there are only xLimit/yLimit unique values
  x <- round(x)
  y <- round(y)

  ## Find non-duplicated elements
  mUniq <- !duplicated(cbind(x, y))

  ## Return values that should be kept
  return(mUniq)
}

## Draw an axis selecting sensible ticks for a date-time values
## Identical to R axis.POSIXct (src/library/graphics/R/datetime.R, GPL license)
##  except where indicated by "SJM:"
## Modified from original to allow requested number of intervals (cIntervals) to
## be specified

DateTimeAxis <- function (side, x, at, cIntervals, format, labels = TRUE, ...)
{
    mat <- missing(at) || is.null(at)
    if(!mat) x <- as.POSIXct(at) else x <- as.POSIXct(x)
    range <- par("usr")[if(side %%2) 1:2 else 3:4]
    ## find out the scale involved
    d <- range[2] - range[1]
    z <- c(range, x[is.finite(x)])
    if(d < 1.1*60) { # seconds
        sc <- 1
        if(missing(format)) format <- "%S"
    } else if (d < 1.1*60*60) { # minutes
        sc <- 60
        if(missing(format)) format <- "%M:%S"
    } else if (d < 1.1*60*60*24) {# hours
        sc <- 60*24
        if(missing(format)) format <- "%H:%M"
    } else if (d < 2*60*60*24) {
        ## SJM: Scale is divided by two
        sc <- 60*24 / 2
        if(missing(format)) format <- "%a %H:%M"
    } else if (d < 7*60*60*24) {# days of a week
        sc <- 60*60*24
        if(missing(format)) format <- "%a"
    } else { # days, up to a couple of months
        sc <- 60*60*24
        if(missing(format)) format <- "%b %d"
    }
    if(d < 60*60*24*50) {
        ## SJM: Pass n (number of intervals) to pretty
        zz <- pretty(z/sc, cIntervals)
        z <- zz*sc
        class(z) <- c("POSIXt", "POSIXct")
        if(sc == 60*60*24) z <- as.POSIXct(round(z, "days"))
    } else if(d < 1.1*60*60*24*365) { # months
        class(z) <- c("POSIXt", "POSIXct")
        zz <- as.POSIXlt(z)
        zz$mday <- 1; zz$isdst <- zz$hour <- zz$min <- zz$sec <- 0
        ## SJM: Pass cIntervals (number of intervals) to pretty
        zz$mon <- pretty(zz$mon, cIntervals)
        m <- length(zz$mon)
        m <- rep.int(zz$year[1], m)
        zz$year <- c(m, m+1)
        z <- as.POSIXct(zz)
        if(missing(format)) format <- "%b"
    } else { # years
        class(z) <- c("POSIXt", "POSIXct")
        zz <- as.POSIXlt(z)
        zz$mday <- 1; zz$isdst <- zz$mon <- zz$hour <- zz$min <- zz$sec <- 0
        ## SJM: Pass n (number of intervals) to pretty
        zz$year <- pretty(zz$year, cIntervals)
        z <- as.POSIXct(zz)
        if(missing(format)) format <- "%Y"
    }
    if(!mat) z <- x[is.finite(x)] # override changes
    z <- z[z >= range[1] & z <= range[2]]
    if (identical(labels, TRUE))
        labels <- format(z, format = format)
    else if (identical(labels, FALSE))
        labels <- rep("", length(z)) # suppress labelling of ticks

    ## SJM: Return format, for axis label
    DrawnAt <- axis(side, at = z, labels = labels, ...)
    return(list(DrawnAt, format))
}

## Draw an axis, similar to default axis(), but with black axis and grey ticks
##
## iSide: Side at which axis is to be drawn (see ?axis)
## x: The data used to select appropriate ticks (only used for POSIXct data)
## cIntervals: The desired number of intervals (only used for POSIXct data)
SimpleAxis <- function(iSide, x = NULL, cIntervals = 5) {
  ## Colour to erase a line with
  coBg <- par("bg")
  if (coBg == "transparent")
    ## Erasing with transparent lines doesn't work. Assume a white background
    coBg <- "white"

  ## Draw the axis and tickmarks in grey
  wDefault <- 0.7 # Width of axis ticks
  if (inherits(x, "POSIXct")) {
    ## Variable is POSIXct, so use appropriate intervals for dates
    tmp <- DateTimeAxis(iSide, x, col = "gray50", lwd = wDefault, cIntervals =
                        cIntervals)
    DrawnAt <- tmp[[1]]
    Format <- tmp[[2]]
  } else {
    ## Otherwise just use round numbers
    DrawnAt <- axis(iSide, col = "gray50", lwd = wDefault)
    Format <- NULL
  }

  ## Erase the grey axis line
  wOver <- 1 # Slightly thicker than the original to ensure cover
  axis(iSide, at = DrawnAt, col = coBg, tcl = 0, lwd = wOver, labels = FALSE)

  ## Draw a black axis line (tcl is ticklength)
  axis(iSide, at = DrawnAt, tcl = 0, lwd = wDefault, labels = FALSE)

  return(list(DrawnAt,Format))
}

