#!/usr/bin/env python

### TempCov: Toolkit for investigating temperature covert channels
### Copyright (C) 2006 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
###
### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; either version 2 of the License, or
### (at your option) any later version.
###
### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.
###
### You should have received a copy of the GNU General Public License along
### with this program; if not, write to the Free Software Foundation, Inc.,
### 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
###
### $Id: probe-receiver.py 3381 2006-12-25 18:07:55Z sjm217 $

import sys; from distutils.util import get_platform
sys.path.append("dpkt-1.6/build/lib/")
sys.path.append("pypcap-1.1/build/lib.%s-%s" % (get_platform(), sys.version[0:3]))

import dpkt, pcap, struct

files={}
prefix= None
pPort = None

class Time_Info:
    def __init__(self):
        self.icmp_timing={}
        self.tcp_timing={}

    def sent_icmp(self, src, ts_sec, ts_usec, i):
        if not self.icmp_timing.has_key(src):
            self.icmp_timing[src]=(None,None,None,None)

        sent, rec, ni, data=self.icmp_timing[src]
        if i!=ni:
            # Last sample was lost, discard it
            sent=(ts_sec,ts_usec)
            rec=None
            ni=i
            data=None
            self.icmp_timing[src]=(sent,rec,ni,data)
        elif rec==None:
            # This should not happen
            #  Two packets have been sent with the same i
            sent=(ts_sec,ts_usec)
            rec=None
            ni=i
            data=None
            self.icmp_timing[src]=(sent,rec,ni,data)
        else: # rec!=None
            # Reply has been received, write both out
            fh=get_fh("icmp", src)
            s_sec=ts_sec
            s_usec=ts_usec
            r_sec=rec[0]
            r_usec=rec[1]
            fh.write("%s %s %s %s %s\n"%(s_sec,s_usec,r_sec,r_usec,data))
            self.icmp_timing[src]=(None,None,None,None)
            
    def rec_icmp(self, src, ts_sec, ts_usec, data, i):
        if not self.icmp_timing.has_key(src):
            self.icmp_timing[src]=(None,None,None,None)

        sent, rec, ni, ndata=self.icmp_timing[src]
        if i!=ni:
            # Last sample was lost, discard it
            sent=None
            rec=(ts_sec,ts_usec)
            ni=i
            ndata=data
            self.icmp_timing[src]=(sent,rec,ni,ndata)
        elif sent==None:
            # This should not happen
            #  Two packets have been received with the same i
            sent=None
            rec=(ts_sec,ts_usec)
            ni=i
            ndata=data
            self.icmp_timing[src]=(sent,rec,ni,ndata)
        else: # sent!=None
            # Request has been received, write both out
            fh=get_fh("icmp", src)
            s_sec=sent[0]
            s_usec=sent[1]
            r_sec=ts_sec
            r_usec=ts_usec
            fh.write("%s %s %s %s %s\n"%(s_sec,s_usec,r_sec,r_usec,data))
            self.icmp_timing[src]=(None,None,None,None)
             
    def sent_tcp(self, src, ts_sec, ts_usec, i):
        if not self.tcp_timing.has_key(src):
            self.tcp_timing[src]=(None,None,None,None)

        sent, rec, ni, data=self.tcp_timing[src]
        if i!=ni:
            # Last sample was lost, discard it
            sent=(ts_sec,ts_usec)
            rec=None
            ni=i
            data=None
            self.tcp_timing[src]=(sent,rec,ni,data)
        elif rec==None:
            # This should not happen
            #  Two packets have been sent with the same i
            sent=(ts_sec,ts_usec)
            rec=None
            ni=i
            data=None
            self.tcp_timing[src]=(sent,rec,ni,data)
        else: # rec!=None
            # Reply has been received, write both out
            fh=get_fh("tcp", src)
            s_sec=ts_sec
            s_usec=ts_usec
            r_sec=rec[0]
            r_usec=rec[1]
            ts=data[0]
            seq=data[1]
            fh.write("%s %s %s %s %s %s\n"%(s_sec,s_usec,r_sec,r_usec,ts,seq))
            self.tcp_timing[src]=(None,None,None,None)
            
    def rec_tcp(self, src, ts_sec, ts_usec, data, i):
        if not self.tcp_timing.has_key(src):
            self.tcp_timing[src]=(None,None,None,None)

        sent, rec, ni, ndata=self.tcp_timing[src]
        if i!=ni:
            # Last sample was lost, discard it
            sent=None
            rec=(ts_sec,ts_usec)
            ni=i
            ndata=data
            self.tcp_timing[src]=(sent,rec,ni,ndata)
        elif sent==None:
            # This should not happen
            #  Two packets have been received with the same i
            sent=None
            rec=(ts_sec,ts_usec)
            ni=i
            ndata=data
            self.tcp_timing[src]=(sent,rec,ni,ndata)
        else: # sent!=None
            # Request has been received, write both out
            fh=get_fh("tcp", src)
            s_sec=sent[0]
            s_usec=sent[1]
            r_sec=ts_sec
            r_usec=ts_usec
            ts=data[0]
            seq=data[1]
            fh.write("%s %s %s %s %s %s\n"%(s_sec,s_usec,r_sec,r_usec,ts,seq))
            self.tcp_timing[src]=(None,None,None,None)

info=Time_Info()

def ipToDotted(ip):
    host=map(str,struct.unpack("BBBB",ip))
    return ".".join(host)

def get_fh(type, src):
    filename=type+"_"+ipToDotted(src)
    fh=files.get(filename, None)
    if not fh:
        # This IP hasn't been seen - create a new file
        fh=file(prefix+filename+".data", "wt")
        files[filename]=fh
        if type=="tcp":
            fh.write("sSec sUsec rSec rUsec ts seq\n")
        else:
            fh.write("sSec sUsec rSec rUsec ts\n")
    return fh

def process_tcp(ts_sec, ts_usec, pkt, src, dst):
    if pkt.flags==18 and pkt.dport==20 and pkt.sport==pPort:
        # SYN ACK
        seq=pkt.seq
        opts=dpkt.tcp.parse_opts(pkt.opts)
        ts="NA"
        for code, data in opts:
            if code==8:
                ts=struct.unpack("!LL",data)[0]
        i=pkt.ack-1
        if i<0:
            i=2*32-1
        info.rec_tcp(src, ts_sec, ts_usec, (ts, seq), i)
    elif pkt.flags==2 and pkt.dport==pPort and pkt.sport==20:
        i=pkt.seq
        info.sent_tcp(dst, ts_sec, ts_usec, i)

def process_icmp(ts_sec, ts_usec, pkt, src, dst):
    if pkt.type==14:
        i=struct.unpack("!H",pkt.data[4:6])[0]
        ts=struct.unpack("!L",pkt.data[16:])[0]
        info.rec_icmp(src, ts_sec, ts_usec, ts, i)
    elif pkt.type==13:
        i=struct.unpack("!H",pkt.data[4:6])[0]
        info.sent_icmp(dst, ts_sec, ts_usec, i)

def process(ts_sec, ts_usec, pkt):
    data=pkt
    if not data.__class__ is dpkt.ethernet.Ethernet:
        return

    data=data.data
    if not data.__class__ is dpkt.ip.IP:
        return

    src=data.src
    dst=data.dst
    data=data.data
    if data.__class__ is dpkt.icmp.ICMP:
        process_icmp(ts_sec, ts_usec, data, src, dst)
    elif data.__class__ is dpkt.tcp.TCP:
        process_tcp(ts_sec, ts_usec, data, src, dst)

def main():
    global prefix, pPort

    if len(sys.argv)!=4:
        print "Usage: probe-receiver PREFIX PORT FILTER"
        sys.exit()
        
    prefix = sys.argv[1]
    pPort = int(sys.argv[2])
    sFilter = sys.argv[3]

    pc = pcap.pcap("eth0",promisc=False)
    pc.setfilter(sFilter)
    decode = { pcap.DLT_LOOP:dpkt.loopback.Loopback,
               pcap.DLT_NULL:dpkt.loopback.Loopback,
               pcap.DLT_EN10MB:dpkt.ethernet.Ethernet }[pc.datalink()]
    try:
        print 'listening on %s: %s' % (pc.name, pc.filter)
        for ts_sec, ts_usec, pkt in pc:
            process(ts_sec,ts_usec,decode(pkt))
    except KeyboardInterrupt:
        nrecv, ndrop, nifdrop = pc.stats()
        print '\n%d packets received by filter' % nrecv
        print '%d packets dropped by kernel' % ndrop

if __name__ == '__main__':
    main()
