/*

Copyright © 2023-24 Sean Holden. All rights reserved.

*/
/*

This file is part of Connect++.

Connect++ is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation, either version 3 of the License, or (at your 
option) any later version.

Connect++ is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
more details.

You should have received a copy of the GNU General Public License along 
with Connect++. If not, see <https://www.gnu.org/licenses/>. 

*/

#include <iostream>
#include <vector>
#include <chrono>
#include <filesystem>
#include <ctime>
#include <cstdlib>
#include <csignal>

#include <boost/program_options.hpp>

#include "StackProver.hpp"
#include "Schedule.hpp"

using std::cout;
using std::endl;
namespace fs = std::filesystem;
namespace chrono = std::chrono;

using namespace boost::spirit;
namespace po = boost::program_options;
using namespace verbose_print;

/*
* Various signal handlers.
*/
void SIGINT_handler(int p) {
  cout << endl << "% SZS status Error for " 
       << params::problem_name 
       << " : terminated by SIGINT." << endl;
  exit(EXIT_FAILURE);
}
void SIGXCPU_handler(int p) {
  cout << endl << "% SZS status Error for "
       << params::problem_name
       << " : terminated by SIGXCPU." << endl;
  exit(EXIT_FAILURE);
}
void SIGALRM_handler(int p) {
  cout << endl << "% SZS status Error for " 
       << params::problem_name 
       << " : terminated by SIGALRM." << endl;
  exit(EXIT_FAILURE);
}

int main(int argc, char** argv) {
  /*
  * Start the timer...
  */
  chrono::steady_clock::time_point startTime;
  startTime = chrono::steady_clock::now();
  /*
  * Set up handlers for signals.
  */
  std::signal(SIGINT, SIGINT_handler);
  std::signal(SIGXCPU, SIGXCPU_handler);
  std::signal(SIGALRM, SIGALRM_handler);
  /*
  * Pick up relevant environment variables for the 
  * TPTP library etc. Some of these may be overidden by a 
  * command line argument later.
  */
  char* tptp_env = getenv("TPTP");
  if (tptp_env != nullptr) {
     params::tptp_path = fs::path(tptp_env);
  }
  char* swi_env = getenv("CONNECTPP_SWI_PATH");
  if (swi_env != nullptr) {
     params::swi_path = fs::path(swi_env);
  }
  char* pwd_env = getenv("PWD");
  if (pwd_env != nullptr) {
     params::pwd_path = fs::path(pwd_env);
  }
  char* path_env = getenv("CONNECTPP_PATH");
  if (path_env != nullptr) {
     params::connectpp_path = fs::path(path_env);
  }
  else {
     params::connectpp_path = params::pwd_path;
  }
  /*
  * Initially, use the Boost Program Options library to decipher
  * what's been added in the command line and deal with it.
  */
  po::options_description main_options("Basic options");
  main_options.add_options()
    ("help,h", 
        "Show this message.")
    ("version", 
        "Show version number.")
    ("input,i", po::value<string>(), 
        "Input filename.")
    ("output,o", po::value<string>(), 
        "Write a short output summary (file, status, outcome, time etc) to the specified file. Using \"default\" places it in \"./output_summary.txt\"" )
    ("no-equality", 
        "Don't add equality axioms. (For example, if they were added elsewhere.)")
    ("equality-last", 
        "Place equality axioms at the end of the matrix, not the beginning")
  //  ("positive", 
  //      "Use DNF/positive representation. Alters equality axioms and choice of start clause(s).")
    ("timeout,t", po::value<int>(), 
        "Timeout in seconds. (Default UINT32_MAX).")
    ("schedule,s",po::value<string>(), 
        "Instead of running with fixed parameters, use a schedule from the specified file. Use value \"default\" for the default schedule. If no --timeout is specified use 600s.")
    ("tptp", po::value<string>(), 
        "Specify a path for the TPTP. Alternative to using the $TPTP environment variable.")
    ("swi", po::value<string>(), 
        "Specify the path for SWI Prolog.  Alternative to using the $CONNECTPP_SWI_PATH environment variable.")
    ("path", po::value<string>(), 
        "Specify the path for the 'prolog' and 'resources' directories. Alternative to using the $CONNECTPP_PATH environment variable.")
    ("reorder,r", po::value<int>(), 
        "Reorder the clauses a specified number of times.")
    ("no-miniscope", 
        "Don't miniscope when converting to CNF.")
    ("show-clauses",  
        "Show translation of first-order formulas to clauses and halt.");

  po::options_description output_options("Output formatting");
  output_options.add_options()
    ("verbosity,v", po::value<int>(), 
        "Set verbosity level (0-5). (Default 1.)")
    ("no-colour", 
        "Don't use colour to highlight output.")
    ("sub-output", 
        "Include substitutions when writing output.")
    ("indent", po::value<int>(), 
        "Set indentation size. (Default 4.)")
    ("out-int", po::value<int>(), 
        "Interval for updating output (default 50,000).");

  po::options_description search_options("Search control");
  search_options.add_options()
    ("all-start", 
        "Use all start clauses. (Overides other start clause options.)")
    ("pos-neg-start", 
        "Use all positive/negative start clauses, according to representation (Default). (Interacts with other start clause options.)")
    ("conjecture-start", 
        "Use all conjecture clauses as start clauses. (Interacts with other start clause options.)")
    ("restrict-start", 
        "Only use one start clause. (Interacts with other start clause options.)")
    ("no-regularity", 
        "Don't use the regularity test.")
    ("all-reductions", 
        "Compute reductions for all literals in the clause, not just the first one.")
    ("all-extensions", 
        "Compute extensions for all literals in the clause, not just the first one.")
    ("all-lemmata", 
        "When using lemmata, consider all literals in the clause, not just the first one.")
    ("no-lemmata", 
        "Do not use lemmata.");

  po::options_description backtrack_options("Backtracking");
  backtrack_options.add_options()
    ("all-backtrack", 
        "Do not limit any backtracking.")
    ("lemmata-backtrack", 
        "Do not limit backtracking for lemmata.")
    ("reduction-backtrack", 
        "Do not limit backtracking for reductions.")
    ("extension-backtrack",
        "Do not limit backtracking for extensions.")
    ("explore-left-trees", 
        "Apply backtracking within left extension trees.")
  //  ("hard-prune", 
  //      "Use the most aggressive possible backtracking limitation heuristic instead of the leanCop-style one.")
    ("complete,c", po::value<int>(), 
        "Start again with a complete search on reaching the specified depth. (Default UINT32_MAX.)");

  po::options_description depth_options("Depth limiting");
  depth_options.add_options()
    ("depth-limit,d", po::value<int>(), 
        "Maximum depth/path length. (Default UINT32_MAX.)")
    ("start-depth", po::value<int>(), 
        "Initial depth/path length. (Default 1.)")
    ("depth-increment", po::value<int>(), 
        "Increment for iterative deepening. (Default 1.)")
    ("limit-by-tree-depth", 
        "Use tree depth (not path length) for deepening.");

  po::options_description proof_options("Proof generation");
  proof_options.add_options()
    ("no-proof", 
        "Don't construct proofs for output.")
  //  ("all-proofs,a", 
  //      "Find all possible proofs. (Not currently implemented.)")
  //  ("num-proofs,n", po::value<uint32_t>(), 
  //      "Limit on number of proofs to find. (Default 1.) (Not currently implemented.)")
    ("prolog", 
        "Generate Prolog format proof and matrix. Overides --no-proof.")
    ("verify", 
        "Generate Prolog format proof and matrix, and run prolog/check_proof to verify. Overides --no-proof."); 

  po::options_description latex_options("LaTeX output");
  latex_options.add_options()
    ("latex", po::value<string>(), 
        "Make a LaTeX file containing a typeset proof. Use \"default\" for \"./latex_proof.tex\" or specify a destination.")
    ("sub-latex", 
        "Include substitutions in the LaTeX proof.")
    ("tiny-latex", 
        "Typeset the latex proof in a tiny font.")
    ("latex-no-matrix", 
        "Don't include the matrix in the LaTeX proof.");
/*
  po::options_description path_options("Path");
  path_options.add_options()
    ("clean-at", po::value<double>(), 
        "Threshold for path cleaning. (Default 0.5) (Not currently implemented.)")
    ("no-clean", 
        "Don't use path cleaning. (Not currently implemented.)");
*/
  po::options_description all_options("Options");
  all_options.add(main_options)
    .add(output_options)
    .add(search_options)
    .add(backtrack_options)
    .add(depth_options)
    .add(proof_options)
    .add(latex_options);
//    .add(path_options);

  po::positional_options_description pos_opts;
  pos_opts.add("input", -1);
  po::variables_map vars_map;
  try {
    po::store(po::command_line_parser(argc, argv).
              options(all_options).
              positional(pos_opts).
              run(), vars_map);
  }
  /*
  * If the command line options are in error then we exit here.
  */
  catch (boost::program_options::error& err) {
      cout << "% SZS status Error for "
           << params::problem_name << " : " << err.what() << endl;
      return EXIT_FAILURE;
  }
  po::notify(vars_map);
  /*
  * We had a correct parse of the command line, so now we deal with 
  * the options.
  */
  if (vars_map.count("help")) {
    cout << all_options << endl;
    return EXIT_SUCCESS;
  }
  if (vars_map.count("version")) {
    cout << "Connect++ Version V0.2.1." << endl;
    cout << "Copyright (C) 2021-24 Sean Holden. All rights reserved." << endl;
    return EXIT_SUCCESS;
  }
  if (vars_map.count("verbosity")) {
    int v = vars_map["verbosity"].as<int>();
    if (v >= 0 && v <= 5) 
      params::verbosity = v;
    else
      cerr << "Verbosity should be between 0 and 5. Using default." << endl;
  }
  VPrint show(params::verbosity);
  if (vars_map.count("sub-output")) {
      params::output_terms_with_substitution = true;
  }
  if (vars_map.count("indent")) {
    int i = vars_map["indent"].as<int>();
    if (i >= 0 && i <= UINT8_MAX)
      params::indent_size = i;
    else
      cerr << "Indent size is negative or too large. Using default." << endl;
  }
  if (vars_map.count("out-int")) {
    int i = vars_map["out-int"].as<int>();
    if (i > 0 && i <= UINT32_MAX)
      params::output_frequency = i;
    else
      cerr << "Interval is negative or too large. Using default." << endl;
  }
  if (vars_map.count("no-equality")) {
    params::add_equality_axioms = false;
  }
  if (vars_map.count("equality-last")) {
    params::equality_axioms_at_start = false;
  }
  if (vars_map.count("timeout")) {
    params::timeout = true;
    int t = vars_map["timeout"].as<int>();
    if (t > 0) {
      if (t >= 10)
        params::timeout_value = t;
      else {
        cerr << "Timeout is too small. Using 10 seconds." << endl;
        params::timeout_value = 10;
      }
    }
    else
      cerr << "Timeout should be positive. Using default." << endl;
  }
  if (vars_map.count("swi")) {
    params::swi_path = vars_map["swi"].as<string>();
  }
  if (vars_map.count("path")) {
    params::connectpp_path = vars_map["path"].as<string>();
  }
  // You should now have all the paths you need, so 
  // finalize them where possible.
  //params::latex_header_file 
  //  = params::connectpp_path / params::latex_header_file;
  //params::latex_footer_file 
  //  = params::connectpp_path / params::latex_footer_file;
  params::LaTeX_proof_path 
    = params::pwd_path / params::LaTeX_proof_path;
  params::Prolog_matrix_path 
    = params::connectpp_path / params::Prolog_matrix_path;
  params::Prolog_proof_path 
    = params::connectpp_path / params::Prolog_proof_path;
  params::output_summary_path 
    = params::pwd_path / params::output_summary_path;
  // We need a couple of temporary file names to write matrix.pl and 
  // proof.pl into, so that if you have a bunch of instances all 
  // runnning together they don't over-write each other's stuff.
  // Unfortunately the function for this actually makes the file, so 
  // delete it immediately in order to have some control over where 
  // it's placed.
  char m_name[] = "matrixXXXXXX";
  mkstemp(m_name);
  fs::remove(m_name);
  params::Prolog_tmp_matrix_path
    = params::connectpp_path / params::Prolog_tmp_matrix_path 
                             / m_name;
  params::Prolog_tmp_matrix_path += ".pl";
  
  char p_name[] = "proofXXXXXX";
  mkstemp(p_name);
  fs::remove(p_name);
  params::Prolog_tmp_proof_path
    = params::connectpp_path / params::Prolog_tmp_proof_path 
                             / p_name;
  params::Prolog_tmp_proof_path += ".pl"; 

  schedule::Schedule schedule;
  if (vars_map.count("output")) {
    params::write_output_summary = true;
    string op = vars_map["output"].as<string>();
    if (op != "default")
      params::output_summary_path = op;
  }
  if (vars_map.count("schedule")) {
    params::use_schedule = true;
    if (!params::timeout) {
      params::timeout = true;
      params::timeout_value = 600;
    }
    params::schedule_path = vars_map["schedule"].as<string>();
    try {
      schedule.read_schedule_from_file(params::schedule_path);
    }
    catch (std::exception& err) {
      cout << "% SZS status Error for "
           << params::problem_name << " : " << err.what() << endl;
      return EXIT_FAILURE;
    }
  }
  /*
  * Note: overides $TPTP environment variable, which is read 
  * earlier.
  */
  if (vars_map.count("tptp")) {
    params::tptp_path = vars_map["tptp"].as<string>();
  }
  if (vars_map.count("reorder")) {
    int n = vars_map["reorder"].as<int>();
    if (n > 0) {
      params::deterministic_reorder = true;
      params::number_of_reorders = n;
    }
    else 
      cerr << "Number of re-orders should be positive. Skipping." << endl;
  }
  if (vars_map.count("no-miniscope")) {
    params::miniscope = false;
  }
  if (vars_map.count("show-clauses")) {
    params::show_clauses = true;
  }
  /*
  if (vars_map.count("positive")) {
    params::positive_representation = true;
  }
  */
  if (vars_map.count("no-colour")) {
    params::use_colours = false;
  }
  /*
  * Set up the path for the input file. A path to the TPTP may 
  * have been set by now either by picking up the $TPTP environment 
  * variable or using a command-line option. Append the file name 
  * to that. This could be improved but seems fine for now.
  */
  fs::path path;
  if (vars_map.count("input")) {
    fs::path initial_path(params::tptp_path);
    path = initial_path.append(vars_map["input"].as<string>());
  }
  else {
    cout << "% SZS status Error for " 
         << params::problem_name << " : no problem specified."  << endl;
    return EXIT_FAILURE;
  }
  params::problem_name = path.stem().string(); 
  /*
  if (vars_map.count("clean-at")) {
    double t = vars_map["clean-at"].as<double>();
    if (t >= 0.0 && t <= 1.0)
      params::path_clean_threshold = t;
    else
      cerr << error_string <<"path clean threshold should be in [0,1]. Using default." << endl;
  }
  if (vars_map.count("no-clean")) {
      params::path_clean = false;
  }
  */
  if (vars_map.count("depth-limit")) {
    int l = vars_map["depth-limit"].as<int>();
    if (l > 1)
      params::depth_limit = l;
    else
      cerr << "Depth limit should be positive. Using default." << endl;
  }
  if (vars_map.count("start-depth")) {
    int l = vars_map["start-depth"].as<int>();
    if (l > 1 && l <= params::depth_limit)
      params::start_depth = l;
    else
      cerr << "Start depth should be positive and bounded by the maximum depth. Using default." << endl;
  }
  if (vars_map.count("depth-increment")) {
    int l = vars_map["depth-increment"].as<int>();
    if (l > 0)
      params::depth_increment = l;
    else
      cerr << "Depth increment should be positive. Using default." << endl;
  }
  if (vars_map.count("limit-by-tree-depth")) {
    params::limit_by_tree_depth = true;
  }
  /**
  * Start options can appear in combinations. The method 
  * StackProver::set_up_start_clauses deals with these. In 
  * summary:
  *
  * - all-start overides everthing.
  *
  * - the remaining three are combined and the method tries 
  *   to do that sensibly. So for example, if you have pos-neg-start 
  *   and restrict-start you should just get a single pos or neg 
  *   clause as a start clause. 
  */
  if (vars_map.count("restrict-start")) {
    params::restrict_start = true;
  }
  if (vars_map.count("pos-neg-start")) {
    params::all_pos_neg_start = true;
  }
  if (vars_map.count("conjecture-start")) {
    params::conjecture_start = true;
  }
  if (vars_map.count("all-start")) {
    show(2, string("--all-start set. This cancels all other start options."), true);
    params::set_all_start();
  }
  /*
  * Careful! It is possible that no start options were set. If that's 
  * the case just set a sensible one.
  */
  if (params::no_start_options()) {
    params::correct_missing_start_options();
  }
  if (vars_map.count("no-regularity")) {
    params::use_regularity_test = false;
  }
  if (vars_map.count("lemmata-backtrack")) {
    params::limit_bt_lemmas = false;
  }
  if (vars_map.count("reduction-backtrack")) {
    params::limit_bt_reductions = false;
  }
  if (vars_map.count("extension-backtrack")) {
    params::limit_bt_extensions = false;
  }
  if (vars_map.count("explore-left-trees")) {
    params::limit_bt_extensions_left_tree = false;
  }
  if (vars_map.count("hard-prune")) {
    params::hard_prune = true;
  }
  if (vars_map.count("all-backtrack")) {
    show(2, string("--all-backtrack set. This prevails over all other backtracking options."), true);
    params::set_all_backtrack();
  }
  if (vars_map.count("all-reductions")) {
    params::limit_reductions = false;
  }
   if (vars_map.count("all-extensions")) {
    params::limit_extensions = false;
  }
  if (vars_map.count("all-lemmata")) {
    params::limit_lemmata = false;
  }
  if (vars_map.count("no-lemmata")) {
    params::use_lemmata = false;
  }
  if (vars_map.count("complete")) {
    int d = vars_map["complete"].as<int>();
    if (d >= params::start_depth)
      params::switch_to_complete = d;
    else {
      cerr << "Switch to complete search must happen after start depth. Using start depth." << endl;
      params::switch_to_complete = params::start_depth;
    }
  }
  if (vars_map.count("no-proof")) {
      params::build_proof = false;
  }
  if (vars_map.count("latex")) {
      params::generate_LaTeX_proof = true;
      string lp = vars_map["latex"].as<string>();
      if (lp != "default")
        params::LaTeX_proof_path = lp;
  }
  if (vars_map.count("sub-latex")) {
      params::sub_LaTeX_proof = true;
  }
  if (vars_map.count("tiny-latex")) {
      params::latex_tiny_proof = true;
  }
  if (vars_map.count("latex-no-matrix")) {
      params::latex_include_matrix = false;
  }
  /*
  if (vars_map.count("all-proofs")) {
      params::all_proofs = true;
  }
  if (vars_map.count("num-proofs")) {
      uint32_t l = vars_map["num-proofs"].as<uint32_t>();
      if (l > 0)
        params::num_proofs = l;
      else
        cerr << error_string << "if you want n proofs, make n>0. Using default." << endl;
  }
  */
  if (vars_map.count("prolog")) {
    params::generate_Prolog_proof = true;
    params::build_proof = true;
  }
  if (vars_map.count("verify")) {
      params::generate_Prolog_proof = true;
      params::verify_proof = true;
      params::build_proof = true;
  }
  // If you're making a prolog proof, make sure any old proof.pl and 
  // matrix.pl get deleted first. 
  if (params::generate_Prolog_proof) {
      fs::remove(params::Prolog_matrix_path);
      fs::remove(params::Prolog_proof_path);
  }
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  /*
  * Now we're ready to start the party.
  */
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  /*
  * First, state the time we're starting at.
  */
  show(1, string("Connect++ started at: "));
  time_t date_and_time;
  time(&date_and_time);
  show(1, asctime(localtime(&date_and_time)));
  /**
  * You'll obviously be needing a prover.
  */
  StackProver prover;
  /**
  * See if you can read the supplied input file.
  */
  show(1, string("Reading from: "));
  show(1, path.c_str(), true);
  bool found_conjecture = false;
  size_t fof_size = 0;
  try {
    prover.read_from_tptp_file(path.c_str(), found_conjecture, fof_size);
  }
  catch (std::exception& err) {
    cout << "% SZS status Error for " 
         << params::problem_name << " : " 
         << err.what() << endl;
    return EXIT_FAILURE;
  }

  chrono::milliseconds parse_time =
    chrono::duration_cast<chrono::milliseconds>(chrono::steady_clock::now() - startTime);
  show(1, string("Total parse and conversion time: "));
  show(1, std::to_string(parse_time.count()));
  show(1, string(" milliseconds."), true);

  prover.set_problem_path(path);
  show(2, string("Starting proof attempt for: "), false);
  show(2, params::problem_name, true);
  /**
  * Now you're ready to try to prove something...
  */
  if (params::verbosity >= 2) {
    prover.show_matrix();
  }
  ProverOutcome prover_outcome;
  ProverOutcome final_outcome;
  final_outcome = ProverOutcome::TimeOut;
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  /*
  * No schedule.
  */
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  if (!params::use_schedule) {
    /**
    * We're not using a schedule, so just run once using the
    * command line options. 
    *
    * Make sure you deal with reordering. 
    */
    if (params::deterministic_reorder) {
      prover.deterministic_reorder(params::number_of_reorders);
    }
    /**
    * You want any timeout to include everything...
    */
    if (params::timeout) {
      chrono::seconds timeout_duration(params::timeout_value);
      prover.set_timeout(startTime + timeout_duration);
    }
    /*
    * Run the prover.
    */
    prover_outcome = prover.prove();
    final_outcome = prover_outcome;
  }
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  /*
  * A schedule is being used.
  */
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  else {
    /**
    * Run using the specified or default schedule. 
    */
    if (!params::timeout) {
      params::timeout = true;
      params::timeout_value = 600;
    }
    /* 
    * Convert to milliseconds and get a standard chunk of time.
    */
    chrono::milliseconds time_chunk((params::timeout_value * 1000) / 100);
    chrono::steady_clock::time_point start_time = startTime;
    chrono::milliseconds chunk_time;
    /*
    * If we're using a schedule, then it's already been read from the relevant 
    * file, so we can start it now.
    */
    size_t schedule_step_number = 1;
    pair<bool, unsigned int> next_schedule = 
              schedule.set_next_schedule();
    if (params::verbosity > 0)
      cout << endl;
    /*
    * Try each step of the schedule in turn.
    */
    while (next_schedule.first) {
      show(1, string("Schedule step "));
      show(1, std::to_string(schedule_step_number));
      show(1, string(": "));
      show(1, schedule.step_to_string(schedule_step_number - 1), true);

      if (next_schedule.second == 0) { 
        prover.set_timeout(startTime + chrono::seconds(params::timeout_value));
      }
      else if (schedule_step_number == 1) {
        prover.set_timeout(startTime + (next_schedule.second * time_chunk));
      }
      else {
        prover.set_timeout(start_time + (next_schedule.second * time_chunk));
      }

      prover_outcome = prover.prove();
      
      if (prover_outcome == ProverOutcome::Valid ||
        prover_outcome == ProverOutcome::Error ||
        prover_outcome == ProverOutcome::False) {
          final_outcome = prover_outcome;
          break;
      }
      
      chunk_time = chrono::duration_cast<chrono::milliseconds>(chrono::steady_clock::now() - start_time);
      
      show.nl(1, 2);
      show(1, string("Timeout after "));
      show(1, std::to_string(chunk_time.count()));
      show(1, string(" milliseconds"), true);

      start_time = chrono::steady_clock::now();
      schedule_step_number++;
      next_schedule = schedule.set_next_schedule();
    }
  }
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  // We're done, so tidy up.
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  prover.show_statistics();
  /**
  * Stop the timer.
  */
  chrono::milliseconds runTime =
    chrono::duration_cast<chrono::milliseconds>(chrono::steady_clock::now() - startTime);
  show(1, string("Total run time: "));
  show(1, std::to_string(runTime.count()));
  show(1, string(" milliseconds."), true);
  /*
   * Display the outcome.
   *
   * TODO - This assumes negative representation. Update to give better 
   * output if positive representation is being used. This of course 
   * depends on whether I decide to reinstate positive representation...
   *
   * Also, note that ContradictoryAxioms is not tested for. In the 
   * current setup Theorem will be reported. 
   */
   string status(prover.get_status());
   string outcome;
   cout << "% SZS status ";
   switch (final_outcome) {
    case ProverOutcome::Valid:
      if (found_conjecture) 
        // Note: ideally would like to deal with 
        // ContradictoryAxioms at some point.
        outcome = "Theorem";
      else
        outcome = "UnSatisfiable";
      break;
    case ProverOutcome::False:
      if (found_conjecture)
        outcome = "CounterSatisfiable";
      else
        outcome = "Satisfiable";
      break;
    case ProverOutcome::PathLenLimit:
      outcome = "GaveUp";
      break;
    case ProverOutcome::Error:
      outcome = "Error";
      break;
    case ProverOutcome::TimeOut:
      outcome = "Timeout";
      break;
    default:
      break;
  }
  cout << outcome << " for " << params::problem_name;
  bool verified = false;
  if (params::verify_proof && final_outcome == ProverOutcome::Valid) {
      std::filesystem::path p(params::swi_path);
      p.append("swipl");
      string command(p.c_str());
      command += " ";
      command += (params::connectpp_path / "prolog/check_proof").c_str();
      command += " ";
      command += params::Prolog_tmp_matrix_path.c_str();
      command += " ";
      command += params::Prolog_tmp_proof_path.c_str();
      command += " >/dev/null";
      int verification_outcome = system(command.c_str());
      if (verification_outcome == 0) {
        verified = true;
        cout << " : Verified";
      }
      else 
        cout << " : FailedVerification";
      // Make sure any old proofXXXXXX.pl and 
      // matrixXXXXXX.pl get deleted first. 
      fs::remove(params::Prolog_tmp_matrix_path);
      fs::remove(params::Prolog_tmp_proof_path);
  }
  cout << endl;
  /*
  * Leave a minimal file with some outcome information.
  */
  if (params::write_output_summary) {
    try {
      std::ofstream out_file(params::output_summary_path);
      out_file << "ProblemName: " << params::problem_name << endl;
      if (found_conjecture) 
        out_file << "ConjectureFound" << endl;
      else  
        out_file << "NoConjectureFound" << endl; 
      if (status == "") 
        status = "NoStatus";
      out_file << "ProblemStatus: " << status << endl;
      out_file << "ProverOutcome: " << outcome << endl;
      if (params::verify_proof && final_outcome == ProverOutcome::Valid) {
        if (verified) 
          out_file << "Verified" << endl;
        else
          out_file << "FailedVerification" << endl; 
      }
      else
        out_file << "NoVerify" << endl;
      out_file << "Time: " << runTime.count() << endl;
      out_file.close();
    }
    catch (std::exception& err) {
      cerr << "Error: can't write output summary." << endl;
    }
  }

  return EXIT_SUCCESS;
}
