/*

Copyright © 2023 Sean Holden. All rights reserved.

*/
/*

This file is part of Connect++.

Connect++ is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation, either version 3 of the License, or (at your 
option) any later version.

Connect++ is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
more details.

You should have received a copy of the GNU General Public License along 
with Connect++. If not, see <https://www.gnu.org/licenses/>. 

*/

#include "FOF.hpp"

VariableIndex* FOF::var_index = nullptr;
FunctionIndex* FOF::fun_index = nullptr;
TermIndex* FOF::term_index = nullptr;

//---------------------------------------------------------------------------
// Simple methods for Variables.
//---------------------------------------------------------------------------
bool FOF::has_free_variable(Variable* v) {
  switch (type) {
    case FOFType::Empty:
      cerr << "Don't do this with an Empty FOF!" << endl;
      return false;
      break;
    case FOFType::Atom:
      return pred.contains_variable(v);
    case FOFType::Neg:
      return (sub_formulas[0].has_free_variable(v));
      break;
    case FOFType::And:
    case FOFType::Or:
      for (size_t i = 0; i < sub_formulas.size(); i++) {
        if (sub_formulas[i].has_free_variable(v)) {
          return true;
        }
      }
      return false;
      break;
    case FOFType::Imp:
    case FOFType::Iff:
      return (sub_formulas[0].has_free_variable(v) ||
              sub_formulas[1].has_free_variable(v));
      break;
    case FOFType::A:
    case FOFType::E:
      if (var == v) 
        return false;
      return sub_formulas[0].has_free_variable(v);
      break;
    default:
      break;
  }
  return false;
}
//---------------------------------------------------------------------------
void FOF::replace_variable(Variable* new_var, Variable* old_var) {
  if (type == FOFType::Empty) {
    cerr << "Don't do this with an Empty FOF!" << endl;
    return;
  }
  if (type == FOFType::Atom) {
    pred.replace_variable(new_var, old_var, term_index);
  }
  else {
    for (FOF& f : sub_formulas)
      f.replace_variable(new_var, old_var);
  }
}
//---------------------------------------------------------------------------
// Skolemization
//---------------------------------------------------------------------------
Term* FOF::make_skolem_function(const vector<Term*>& args) {
  Function* sk = fun_index->make_skolem_function(args.size());
  Term* sk_t = term_index->add_function_term(sk, args);
  return sk_t;
}
//---------------------------------------------------------------------------
void FOF::replace_variable_with_term(Term* new_term, Variable* old_var) {
  if (type == FOFType::Empty) {
    cerr << "Don't do this with an Empty FOF!" << endl;
    return;
  }
  if (type == FOFType::Atom) {
    pred.replace_variable_with_term(new_term, old_var, term_index);
  }
  else {
    for (FOF& f : sub_formulas)
      f.replace_variable_with_term(new_term, old_var);
  }
}
//---------------------------------------------------------------------------
void FOF::skolemize_main(vector<Term*> skolem_arguments) {
  Term* sk_t;
  FOF f(FOFType::Atom);
  switch (type) {
    case FOFType::Atom:
      break;
    case FOFType::Neg:
      sub_formulas[0].skolemize_main(skolem_arguments);
      break;
    case FOFType::And:
    case FOFType::Or:
    case FOFType::Imp:
    case FOFType::Iff:
      for (FOF& g : sub_formulas)
        g.skolemize_main(skolem_arguments);
      break;
    case FOFType::A:
      skolem_arguments.push_back(term_index->add_variable_term(var));
      sub_formulas[0].skolemize_main(skolem_arguments);
      break;
    case FOFType::E:
      sk_t = make_skolem_function(skolem_arguments);
      sub_formulas[0].replace_variable_with_term(sk_t, var);
      sub_formulas[0].skolemize_main(skolem_arguments);
      f = sub_formulas[0];
      *this = f;
      break;
    default:
      break;
    }
}
//---------------------------------------------------------------------------
void FOF::skolemize() {
  vector<Term*> skolem_arguments;
  skolemize_main(skolem_arguments);
}
//---------------------------------------------------------------------------
// Miniscoping
//---------------------------------------------------------------------------
void FOF::miniscope_split(Variable* v, 
                          vector<FOF>& free, 
                          vector<FOF>& absent) {
  vector<FOF>& subs = sub_formulas[0].sub_formulas;
  for (size_t i = 0; i < subs.size(); i++) {
    FOF f(subs[i]);
    if (f.has_free_variable(v))
      free.push_back(f);
    else
      absent.push_back(f);
  }
}
//---------------------------------------------------------------------------
void FOF::miniscope_all() {
  for (size_t i = 0; i < sub_formulas.size(); i++ ) {
    sub_formulas[i].miniscope();
  }
}
//---------------------------------------------------------------------------
void FOF::miniscope() {
  vector<FOF> free;
  vector<FOF> absent;
  FOF new_sub(FOFType::Empty);
  vector<FOF> new_subs;
  Variable* new_var;
  switch (type) {
    case FOFType::Empty:
      cerr << "Don't do this with an empty formula" << endl;
      break;
    case FOFType::Atom:
    case FOFType::Neg:
      break;
    case FOFType::And:
    case FOFType::Or:
      miniscope_all();
      break;
    case FOFType::Imp:
    case FOFType::Iff:
      cerr << "Don't do this unless you've removed -> and <->" << endl;
      break;
    // Remember you're only dealing with stuff in NNF. You 
    // do however have to handle nested quantifiers.
    case FOFType::A:
      sub_formulas[0].miniscope();
      if (!(sub_formulas[0].type == FOFType::And ||
          sub_formulas[0].type == FOFType::Or)) {
        return;
      }
      miniscope_split(var, free, absent);
      if (sub_formulas[0].type == FOFType::And) {
        type = FOFType::And;
        // If everything is bound then make a new forall everywhere
        // with a new variable.
        if (free.size() == sub_formulas[0].sub_formulas.size()) {
          for (size_t i = 0; i < free.size(); i++) {
            new_var = var_index->add_anon_var();
            new_sub = free[i];
            new_sub.replace_variable(new_var, var);
            new_sub = FOF::make_forall(new_sub, new_var);
            new_subs.push_back(new_sub);
          }
          sub_formulas = new_subs;
        }
        // If any subformula doesn't have the variable then just
        // move the forall in.
        else {
          sub_formulas = absent;
          if (free.size() == 1)
            new_sub = free[0];
          else
            new_sub = FOF::make_and(free);
          new_sub = FOF::make_forall(new_sub, var);
          sub_formulas.push_back(new_sub);
        }
        var = nullptr;
        miniscope_all();
        return;
      }
      if (sub_formulas[0].type == FOFType::Or) {
        // If eveything is bound there's nothing to do.
        if (free.size() != sub_formulas[0].sub_formulas.size()) {
          type = FOFType::Or;
          sub_formulas = absent;
          if (free.size() == 1)
            new_sub = free[0];
          else
            new_sub = FOF::make_or(free);
          new_sub = FOF::make_forall(new_sub, var);
          sub_formulas.push_back(new_sub);
          var = nullptr;
          cout << "! " << *this << endl;
          miniscope_all();
        }
        return;
      }
      break;
    case FOFType::E:
      sub_formulas[0].miniscope();
      if (!(sub_formulas[0].type == FOFType::And ||
          sub_formulas[0].type == FOFType::Or)) {
        return;
      }
      miniscope_split(var, free, absent);
      if (sub_formulas[0].type == FOFType::Or) {
        type = FOFType::Or;
        // If everything is bound then make a new exists everywhere
        // with a new variable.
        if (free.size() == sub_formulas[0].sub_formulas.size()) {
          for (size_t i = 0; i < free.size(); i++) {
            new_var = var_index->add_anon_var();
            new_sub = free[i];
            new_sub.replace_variable(new_var, var);
            new_sub = FOF::make_exists(new_sub, new_var);
            new_subs.push_back(new_sub);
          }
          sub_formulas = new_subs;
        }
        // If any subformula doesn't have the variable then just
        // move the forall in.
        else {
          sub_formulas = absent;
          if (free.size() == 1)
            new_sub = free[0];
          else
            new_sub = FOF::make_or(free);
          new_sub = FOF::make_exists(new_sub, var);
          sub_formulas.push_back(new_sub);
        }
        var = nullptr;
        miniscope_all();
        return;
      }
      if (sub_formulas[0].type == FOFType::And) {
        // If eveything is bound there's nothing to do.
        if (free.size() != sub_formulas[0].sub_formulas.size()) {
          type = FOFType::And;
          sub_formulas = absent;
           if (free.size() == 1)
            new_sub = free[0];
          else
            new_sub = FOF::make_and(free);
          new_sub = FOF::make_exists(new_sub, var);
          sub_formulas.push_back(new_sub);
          var = nullptr;
          miniscope_all();
        }
        return;
      }
      break;
    default:
      break;
    }
}
//---------------------------------------------------------------------------
// Constructors.
//---------------------------------------------------------------------------
FOF::FOF(const Literal& lit)
: sub_formulas()
, var(nullptr)
{
  if (lit.is_positive()) {
    type = FOFType::Atom;
    pred = lit;
  }
  else {
    pred.clear();
    type = FOFType::Neg;
    Literal lit_copy = lit;
    lit_copy.make_positive();
    FOF f(lit_copy);
    sub_formulas.push_back(f);
  }
}
//---------------------------------------------------------------------------
FOF::FOF(FOFType t, const vector<FOF>& sf, Variable* v)
: type(t)
, pred()
, sub_formulas(sf)
, var(v)
{
  switch (t) {
      case FOFType::Atom:
        cerr << "Stop it! You can't make an atom with this constructor!" << endl;
        break;
      case FOFType::Neg:
      case FOFType::And:
      case FOFType::Or:
      case FOFType::Imp:
      case FOFType::Iff:
        // Easy as the work gets done above.
        if (var != nullptr) {
          cerr << "Are you sure you want to associate a variable with something that's not a quantifier?" << endl;
          var = nullptr;
        }
        break;
      case FOFType::A:
      case FOFType::E:
        if (sf.size() != 1) {
          cerr << "Careful! A quantifier should only have one subformula." << endl;
        }
        if (v == nullptr) {
          cerr << "Careful! A quantifier needs a variable." << endl;
        }
        break;
      default:
        break;
  }
}
//---------------------------------------------------------------------------
// Basic tests.
//---------------------------------------------------------------------------
bool FOF::is_literal() const {
  return (type == FOFType::Atom) || 
    (type == FOFType::Neg && sub_formulas[0].type == FOFType::Atom);
}
//---------------------------------------------------------------------------
bool FOF::is_clause() const {
  if (is_literal()) {
    return true;
  }
  if (!(type == FOFType::Or)) {
    return false;
  }
  for (size_t i = 0; i < sub_formulas.size(); i++) {
    if (!sub_formulas[i].is_literal()) {
      return false;
    }
  } 
  return true;
}
//---------------------------------------------------------------------------
// Standard simplifications.
//---------------------------------------------------------------------------
void FOF::remove_negation() {
  if (type == FOFType::Neg) {
    FOF f = sub_formulas[0];
    *this = f;
  }
}
//---------------------------------------------------------------------------
void FOF::negate() {
  if (type == FOFType::Empty) {
    cerr << "Don't do this with an Empty FOF!" << endl;
    return;
  }
  if (type == FOFType::Imp) {
    type = FOFType::And;
    sub_formulas[1].negate();
  }
  else if (type == FOFType::Neg) {
    remove_negation();
  }
  else {
    FOF f = *this;
    type = FOFType::Neg;
    pred.clear();
    var = nullptr;
    sub_formulas.clear();
    sub_formulas.push_back(f);
  }
}
//---------------------------------------------------------------------------
void FOF::remove_iff() {
  if (type == FOFType::Empty) {
    cerr << "Don't do this with an Empty FOF!" << endl;
    return;
  }
  // End of recursion.
  if (type == FOFType::Atom)
    return;
  // Do the relevant transformation.
  if (type == FOFType::Iff) {
    FOF lhs = *this;
    FOF rhs = *this;
    lhs.type = FOFType::Imp;
    rhs.type = FOFType::Imp;
    FOF store = rhs.sub_formulas[1];
    rhs.sub_formulas[1] = rhs.sub_formulas[0];
    rhs.sub_formulas[0] = store;
    type = FOFType::And;
    sub_formulas.clear();
    sub_formulas.push_back(lhs);
    sub_formulas.push_back(rhs);
  }
  // Finish the job. Also applies the transformation to other FOF types.
  for (FOF& f : sub_formulas)
    f.remove_iff();
}
//---------------------------------------------------------------------------
void FOF::remove_imp() {
  if (type == FOFType::Empty) {
    cerr << "Don't do this with an Empty FOF!" << endl;
    return;
  }
  // End of recursion.
  if (type == FOFType::Atom)
    return;
  // Do the relevant transformation.
  if (type == FOFType::Imp) {
    type = FOFType::Or;
    sub_formulas[0].negate();
  }
  // Finish the job. Also applies the transformation to other FOF types.
  for (FOF& f : sub_formulas)
    f.remove_imp();
}
//---------------------------------------------------------------------------
void FOF::push_negs() {
  switch (type) {
    case FOFType::Neg:
      switch (sub_formulas[0].type) {
          case FOFType::Neg:
            remove_negation();
            remove_negation();
            push_negs();
            break;
          case FOFType::Atom:
            break;
          case FOFType::And:
            remove_negation();
            type = FOFType::Or;
            for (FOF& f : sub_formulas) {
              f.negate();
              f.push_negs();
            }
            break;
          case FOFType::Or:
            remove_negation();
            type = FOFType::And;
            for (FOF& f : sub_formulas) {
              f.negate();
              f.push_negs();
            }
            break;
          case FOFType::A:
            remove_negation();
            type = FOFType::E;
            sub_formulas[0].negate();
            sub_formulas[0].push_negs();
            break;
          case FOFType::E:
            remove_negation();
            type = FOFType::A;
            sub_formulas[0].negate();
            sub_formulas[0].push_negs();
            break;
          case FOFType::Imp:
          case FOFType::Iff:
            cerr << "Stop it!!! You don't push negations through -> or <->." << endl;
            break;
          default:
            break;
      }
      break;
    case FOFType::Atom:
      break;
    case FOFType::And:
    case FOFType::Or:
    case FOFType::A:
    case FOFType::E:
      for (FOF& f : sub_formulas)
        f.push_negs();
      break;
    case FOFType::Imp:
    case FOFType::Iff:
      cerr << "Stop it!!! You don't push negations through -> or <->." << endl;
      break;
    default:
      break;
  }
}
//---------------------------------------------------------------------------
void FOF::make_unique_bound_variables() {
  Variable* new_var;
  switch (type) {
    case FOFType::Atom:
      break;
    case FOFType::Neg:
    case FOFType::And:
    case FOFType::Or:
    case FOFType::Imp:
    case FOFType::Iff:
      for (FOF& sf : sub_formulas)
        sf.make_unique_bound_variables();
      break;
    case FOFType::A:
    case FOFType::E:
      sub_formulas[0].make_unique_bound_variables();
      new_var = var_index->add_anon_var();
      sub_formulas[0].replace_variable(new_var, var);
      var = new_var;
      break;
    default:
      break;
    }
}
//---------------------------------------------------------------------------
void FOF::convert_to_nnf() {
  if (type == FOFType::Empty) {
    cerr << "Don't do this with an Empty FOF!" << endl;
    return;
  }
  remove_iff();
  remove_imp();
  push_negs();
}
//---------------------------------------------------------------------------
void FOF::convert_to_cnf() {
  if (type == FOFType::Empty) {
    cerr << "Don't do this with an Empty FOF!" << endl;
    return;
  }
  convert_to_nnf();

}
//---------------------------------------------------------------------------
void FOF::remove_universal_quantifiers() {
  FOF f(FOFType::Atom);
  switch (type) {
    case FOFType::Atom:
      break;
    case FOFType::Neg:
      sub_formulas[0].remove_universal_quantifiers();
      break;
    case FOFType::And:
    case FOFType::Or:
    case FOFType::Imp:
    case FOFType::Iff:
      for (FOF& g : sub_formulas)
        g.remove_universal_quantifiers();
      break;
    case FOFType::A:
      sub_formulas[0].remove_universal_quantifiers();
      f = sub_formulas[0];
      *this = f;
      break;
    case FOFType::E:
      sub_formulas[0].remove_universal_quantifiers();
      break;
    default:
      break;
    }
}
//---------------------------------------------------------------------------
void FOF::to_Literal(Literal& new_lit) const {
    if (type == FOFType::Atom) {
      new_lit = pred;
    }
    else {
      new_lit = sub_formulas[0].pred;
      new_lit.make_negative();
    }
}
//---------------------------------------------------------------------------
void FOF::to_clause(Clause& c) const {
  c.clear();
  Literal new_lit;
  if (is_literal()) {
    to_Literal(new_lit);
    c.add_lit(new_lit);
    return;
  }
  for (size_t i = 0; i < sub_formulas.size(); i++) {
    sub_formulas[i].to_Literal(new_lit);
    c.add_lit(new_lit);
  }
}
//---------------------------------------------------------------------------
void FOF::to_clauses(vector<Clause>& cs) const {
  cs.clear();
  if (is_literal()) {
    Literal new_lit;
    to_Literal(new_lit);
    Clause new_c;
    new_c.add_lit(new_lit);
    cs.push_back(new_c);
    return;
  } 
  if (is_clause()) {
    Clause new_c;
    to_clause(new_c);
    cs.push_back(new_c);
    return;
  }
  if (type == FOFType::And) {
    for (size_t i = 0; i < sub_formulas.size(); i++) {
      if (!sub_formulas[i].is_clause()) {
        cerr << "Your FOF is not in the form of a CNF" << endl;
        continue;
      }
      Clause new_clause;
      sub_formulas[i].to_clause(new_clause);
      cs.push_back(new_clause);
    }
    return;
  } 
  cerr << "You can only make a CNF from an FOF of a sensible form" << endl;
}
//---------------------------------------------------------------------------
bool FOF::has_and() const {
  for (size_t i = 0; i < sub_formulas.size(); i++) {
    if (sub_formulas[i].type == FOFType::And) {
      return true;
    }
  }
  return false;
}
//---------------------------------------------------------------------------
void FOF::collect_ors() {
  vector<FOF> ors;
  vector<FOF> ands;
  for (size_t i = 0; i < sub_formulas.size(); i++) {
    FOF f = sub_formulas[i];
    if (f.type == FOFType::Or || f.is_literal()) 
      ors.push_back(f);
    else 
      ands.push_back(f);
  }
  if (ors.size() > 0) {
    FOF f = ors[0];
    if (ors.size() > 1) 
      f = FOF::make_or(ors);
    ands.push_back(f);
  }
  sub_formulas = ands; 
}
//---------------------------------------------------------------------------
void FOF::distribute_or() {
  if (is_literal()) {
    return;
  }
  // ANDs are straightforward.
  if (type == FOFType::And) {
    for (size_t i = 0; i < sub_formulas.size(); i++) {
      sub_formulas[i].distribute_or();    
    }
    return;
  }
  // You must have an OR, so distribute as needed.
  // If there's no AND at this level then 
  // distribute in the subtrees.
  if (!has_and()) {
    for (size_t i = 0; i < sub_formulas.size(); i++) {
      sub_formulas[i].distribute_or();    
    }
  }
  // You may now have ANDs that have been raised in the 
  // tree, so a step is needed to finish off;
  if (has_and()) {
    // There's an AND, so start by arranging to have only a 
    // single top-level OR.
    collect_ors();
    // Now you have a single thing that looks like 
    // OR( AND(...), AND(), ..., OR() ) with possibly 
    // a literal or nothing on the right.
    while (sub_formulas.size() > 1) {
      size_t n = sub_formulas.size();
      // Each step here merges two subformulas into one.
      vector<FOF> new_sfs1;
      FOF f1 = sub_formulas[n-2];
      FOF f2 = sub_formulas[n-1];
      for (size_t i = 0; i < f1.sub_formulas.size(); i++) {
        vector<FOF> new_sfs2;
        new_sfs2.push_back(f1.sub_formulas[i]);
        new_sfs2.push_back(f2);
        FOF new_f = FOF::make_or(new_sfs2);
        new_sfs1.push_back(new_f);
      }
      sub_formulas[n-2] = FOF::make_and(new_sfs1);
      sub_formulas.pop_back();
    }
    // Now you just have a single AND.
    type = FOFType::And;
    FOF f = sub_formulas[0];
    sub_formulas.clear();
    sub_formulas.push_back(f);
    for (size_t i = 0; i < sub_formulas.size(); i++) {
      sub_formulas[i].distribute_or();    
    }
  } 
}
//---------------------------------------------------------------------------
void FOF::flatten_ands() {
  if (type != FOFType::And) {
    cerr << "You can only flatten a conjunction" << endl;
    return;
  }
  vector<FOF> new_sub_formulas;
  for (size_t i = 0; i < sub_formulas.size(); i++) {
    FOF f = sub_formulas[i];
    if (f.is_literal() ||
        f.type == FOFType::Or) {
          new_sub_formulas.push_back(f);
        }
    else if (f.type == FOFType::And) {
      for (size_t j = 0; j < f.sub_formulas.size(); j++) {
        new_sub_formulas.push_back(f.sub_formulas[j]);
      }
    }
    else {
      cerr << "Why are you flattening something with non-CNFs?" << endl;
      new_sub_formulas.push_back(f);
    }
  }
  sub_formulas = new_sub_formulas;
}
//---------------------------------------------------------------------------
void FOF::flatten() {
  if (is_literal())
    return;
  // OR is easy as everything under it is an OR or a literal.
  if (type == FOFType::Or) {
    vector<FOF> new_subs;
    for (size_t i = 0; i < sub_formulas.size(); i++) {
      FOF f = sub_formulas[i];
      f.flatten();
      // Now the subformula is a literal or a disjunctions of literals.
      if (f.is_literal())
        new_subs.push_back(f);
      else {
        for (size_t j = 0; j < f.sub_formulas.size(); j++) {
          new_subs.push_back(f.sub_formulas[j]);
        }
      }
    }
    sub_formulas = new_subs;
    return;
  }
  // You have an AND.
  for (int i = 0; i < sub_formulas.size(); i++) {
    sub_formulas[i].flatten();
    // Now the subformula is a CNF, a disjunction of literals, or 
    // a literal.
  }
  flatten_ands();
}
//---------------------------------------------------------------------------
string FOF::to_string() const {
  string result;
  size_t s = sub_formulas.size();;
  size_t i = 1;
  switch (type) {
    case FOFType::Atom:
      result = pred.to_string();
      break;
    case FOFType::Neg:
      result = unicode_symbols::LogSym::neg;
      result += sub_formulas[0].to_string();
      break;
    case FOFType::And:
      result = "( ";
      for (const FOF& g : sub_formulas) {
        result += g.to_string();
        if (i < s) {
         result += " ";
          result += unicode_symbols::LogSym::and_sym;
          result += " ";
        }
        i++;
      }
      result += " )";
      break;
    case FOFType::Or:
      result = "( ";
      for (const FOF& g : sub_formulas) {
        result += g.to_string();
        if (i < s) {
            result += " ";
            result += unicode_symbols::LogSym::or_sym;
            result += " ";
        }
        i++;
      }
      result += " )";
      break;
    case FOFType::Imp:
      result =  sub_formulas[0].to_string();
      result += " ";
      result += unicode_symbols::LogSym::ifthen;
      result += " ";
      result += sub_formulas[1].to_string();
      break;
    case FOFType::Iff:
      result = sub_formulas[0].to_string();
      result += " ";
      result += unicode_symbols::LogSym::iff;
      result += " ";
      result += sub_formulas[1].to_string();
      break;
    case FOFType::A:
      result = unicode_symbols::LogSym::forall;
      result += var->to_string();
      result += " . [ ";
      result += sub_formulas[0].to_string();
      result += " ]";
      break;
    case FOFType::E:
      result = unicode_symbols::LogSym::exists;
      result += var->to_string();
      result += " . [ ";
      result += sub_formulas[0].to_string();
      result += " ]";
      break;
    default:
      break;
  }
  return result;
}
//---------------------------------------------------------------------------
ostream& operator<<(ostream& out, const FOF& f) {
  out << f.to_string();
  return out;
}