(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

(*
load "pred_setTheory";
load "ConseqConv";
load "utilLib";
load "x86_event_monadTheory";
load "x86_axiomatic_modelTheory";
load "x86_decoderTheory";
load "x86_event_opsemTheory";
*)

open HolKernel boolLib Parse bossLib pairLib;
open pred_setTheory computeLib ConseqConv;
open utilLib utilTheory;
open x86_event_monadTheory x86_axiomatic_modelTheory x86_astTheory;
open x86_event_opsemTheory x86_typesTheory x86_decoderTheory;
(* open HolDoc; *)

val _ = new_theory "x86_event_opsem_wf";

val _ = Parse.hide "iiid";

val lem1 = Q.prove (
`!x y. 
  (x, y) IN sTC (r UNION r' UNION {(e1,e2) | e1 IN maximal_elements s r /\ e2 IN minimal_elements s' r'}) ==>
  DISJOINT s s' /\
  DOM r SUBSET s /\
  RANGE r SUBSET s /\
  DOM r' SUBSET s' /\
  RANGE r' SUBSET s' /\
  x IN s' ==>
  (x, y) IN sTC r'`,
HO_MATCH_MP_TAC sTC_ind THEN
SRW_TAC [] [] THENL
[IMP_RES_TAC sTC_DOM_RNG_thm THEN
     FULL_SIMP_TAC  (srw_ss()) [SUBSET_DEF, DOM_def, RANGE_def, DISJOINT_DEF, EXTENSION] THEN
     METIS_TAC [],
 METIS_TAC [sTC_rules],
 FULL_SIMP_TAC  (srw_ss()) [maximal_elements_def, DISJOINT_DEF, EXTENSION] THEN
     METIS_TAC [],
 `(x, x') IN sTC r'` by METIS_TAC [] THEN
     IMP_RES_TAC sTC_DOM_RNG_thm THEN
     FULL_SIMP_TAC  (srw_ss()) [SUBSET_DEF] THEN
     METIS_TAC [sTC_rules]]);

val lem2 = Q.prove (
`!x y. 
  (x, y) IN sTC (r UNION r' UNION {(e1,e2) | e1 IN maximal_elements s r /\ e2 IN minimal_elements s' r'}) ==>
  DISJOINT s s' /\
  DOM r SUBSET s /\
  RANGE r SUBSET s /\
  DOM r' SUBSET s' /\
  RANGE r' SUBSET s' /\
  y IN s ==>
  (x, y) IN sTC r`,
HO_MATCH_MP_TAC sTC_ind THEN
SRW_TAC [] [] THENL
[METIS_TAC [sTC_rules],
 IMP_RES_TAC sTC_DOM_RNG_thm THEN
     FULL_SIMP_TAC  (srw_ss()) [SUBSET_DEF, DOM_def, RANGE_def, DISJOINT_DEF, EXTENSION] THEN
     METIS_TAC [],
 FULL_SIMP_TAC  (srw_ss()) [minimal_elements_def, DISJOINT_DEF, EXTENSION] THEN
     METIS_TAC [],
 `(x', y) IN sTC r` by METIS_TAC [] THEN
     IMP_RES_TAC sTC_DOM_RNG_thm THEN
     FULL_SIMP_TAC  (srw_ss()) [SUBSET_DEF] THEN
     METIS_TAC [sTC_rules]]);

val acyclic_seq_union_lem = Q.prove (
`!s s' r r'. 
  DISJOINT s s' /\
  DOM r SUBSET s /\
  RANGE r SUBSET s /\
  DOM r' SUBSET s' /\
  RANGE r' SUBSET s' /\
  acyclic r /\
  acyclic r'
  ==>
  acyclic (r UNION r' UNION
           {(e1,e2) | e1 IN maximal_elements s r /\ e2 IN minimal_elements s' r'})`,
SRW_TAC [] [acyclic_def] THEN
CCONTR_TAC THEN
FULL_SIMP_TAC (srw_ss()) [] THEN
Cases_on `x IN s` THEN1
METIS_TAC [lem2] THEN
Cases_on `x IN s'` THEN1
METIS_TAC [lem1] THEN
IMP_RES_TAC sTC_DOM_RNG_thm THEN
FULL_SIMP_TAC (srw_ss()) [DOM_def, RANGE_def, SUBSET_DEF, maximal_elements_def] THEN
METIS_TAC []);

val LAMBDA_PROD2 = Q.prove (
`!f. (\(x, y). f x y) = (\p. f (FST p) (SND p))`,
RWTAC [FUN_EQ_THM] THEN
RWTAC [FUN_EQ_THM] THEN
Cases_on `p` THEN 
RWTAC []);

val inM_def = Define `
  inM e m = ?eiid_start eiid_next x E. (eiid_next, x, E) IN m eiid_start /\ e IN E.events`;

val if_inM_thm = Q.prove (
`!b m m' e.
  inM e (if b then m else m')
  ==>
  (b /\ inM e m) \/
  (~b /\ inM e m')`,
RWTAC []);

val mapT_inM_thm = Q.prove (
`!f m e. inM e (mapT_ev f m) ==> inM e m`,
RWTAC [inM_def, mapT_ev_def, LET_THM] THEN
METIS_TAC []);

val choiceT_inM_thm = Q.prove (
`!m m' e. inM e (choiceT_ev m m') ==> inM e m \/ inM e m'`,
RWTAC [choiceT_ev_def, inM_def] THEN
METIS_TAC []);

val constT_inM_thm = Q.prove (
`!x e. inM e (constT x) ==> F`,
RWTAC [inM_def, constT_def, constT_ev_def, event_structure_empty_def] THEN
FSTAC []);

val addT_inM_thm = Q.prove (
`!x s e. inM e (addT x s) ==> inM e s`,
RWTAC [addT_def, addT_ev_def, inM_def, LET_THM] THEN
Cases_on `x''` THEN
Cases_on `r` THEN
FSTAC [] THEN
RWTAC [] THEN
METIS_TAC []);

val failureT_inM_thm = Q.prove (
`!e. inM e failureT ==> F`,
RWTAC [inM_def, failureT_def, failureT_ev_def]);

val seqT_inM_thm = Q.prove (
`!m f e. inM e (seqT m f) ==> inM e m \/ (?x. inM e (f x))`,
RWTAC [inM_def, seqT_def, seqT_ev_def, LET_THM, event_structure_seq_union_def] THEN
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN
METIS_TAC []);

val parT_inM_thm = Q.prove (
`!m m' e. inM e (parT m m') ==> inM e m \/ inM e m'`,
RWTAC [inM_def, parT_def, parT_ev_def, LET_THM, event_structure_union_def] THEN
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN
METIS_TAC []);

(* Cut and pasted from above *)
val parT_unit_inM_thm = Q.prove (
`!m m' e. inM e (parT_unit m m') ==> inM e m \/ inM e m'`,
RWTAC [inM_def, parT_unit_def, parT_unit_ev_def, LET_THM, event_structure_union_def] THEN
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN
METIS_TAC []);

val write_reg_inM_thm = Q.prove (
`!ii r x e.
  inM e (write_reg ii r x) 
  ==> 
  (loc e = SOME (Location_reg ii.proc (Reg32 r))) /\
  (e.action = Access W (Location_reg ii.proc (Reg32 r)) x)`,
RWTAC [inM_def, write_reg_def, write_reg_ev_def, write_location_ev_def,
       loc_def] THEN
FSTAC []);

val read_reg_inM_thm = Q.prove (
`!ii r e.
  inM e (read_reg ii r) 
  ==> 
  (loc e = SOME (Location_reg ii.proc (Reg32 r))) /\
  (?v. e.action = Access R (Location_reg ii.proc (Reg32 r)) v)`,
RWTAC [inM_def, read_reg_def, read_reg_ev_def, read_location_ev_def,
       loc_def] THEN
FSTAC []);

val write_eip_inM_thm = Q.prove (
`!ii x e.
  inM e (write_eip ii x) 
  ==> 
  (loc e = SOME (Location_reg ii.proc RegEIP)) /\
  (e.action = Access W (Location_reg ii.proc RegEIP) x)`,
RWTAC [inM_def, write_eip_def, write_eip_ev_def, write_location_ev_def,
       loc_def] THEN
FSTAC []);

val read_eip_inM_thm = Q.prove (
`!ii e.
  inM e (read_eip ii) 
  ==> 
  (loc e = SOME (Location_reg ii.proc RegEIP)) /\
  (?v. e.action = Access R (Location_reg ii.proc RegEIP) v)`,
RWTAC [inM_def, read_eip_def, read_eip_ev_def, read_location_ev_def,
       loc_def] THEN
FSTAC []);

val write_eflag_inM_thm = Q.prove (
`!ii f bo e.
  inM e (write_eflag ii f bo)
  ==>
  (loc e = SOME (Location_reg ii.proc (Reg1 f))) /\
  (?v. e.action = Access W (Location_reg ii.proc (Reg1 f)) v)`,
Cases_on `bo` THEN
RWTAC [write_eflag_def, write_eflag_ev_def, write_location_ev_def, loc_def] THEN
IMP_RES_TAC choiceT_inM_thm THEN
FSTAC [inM_def] THEN
RWTAC [] THEN 
FSTAC [] THEN
RWTAC []);

val read_eflag_inM_thm = Q.prove (
`!ii f e.
  inM e (read_eflag ii f) 
  ==>
  (loc e = SOME (Location_reg ii.proc (Reg1 f))) /\
  (?v. e.action = Access R (Location_reg ii.proc (Reg1 f)) v)`,
RWTAC [read_eflag_def, read_eflag_ev_def, read_location_ev_def, loc_def] THEN
IMP_RES_TAC mapT_inM_thm THEN
FSTAC [inM_def] THEN
RWTAC [] THEN 
FSTAC [] THEN
RWTAC []);

val write_m32_inM_thm = Q.prove (
`!ii a x e.
  inM e (write_m32 ii a x)
  ==>
  (loc e = SOME (Location_mem a)) /\
  (e.action = Access W (Location_mem a) x)`,
RWTAC [write_m32_def, write_m32_ev_def, write_location_ev_def,
       inM_def, loc_def, failureT_ev_def] THEN
FSTAC []);

val read_m32_inM_thm = Q.prove (
`!ii a e.
  inM e (read_m32 ii a)
  ==>
  (loc e = SOME (Location_mem a)) /\
  (?v. e.action = Access R (Location_mem a) v)`,
RWTAC [read_m32_def, read_m32_ev_def, read_location_ev_def,
       inM_def, loc_def, failureT_ev_def] THEN
FSTAC []);

val ea_Xr_inM_thm = Q.prove (
`!r e. inM e (ea_Xr r) ==> F`,
RWTAC [ea_Xr_def, constT_inM_thm]);

val ea_Xi_inM_thm = Q.prove (
`!i e. inM e (ea_Xi i) ==> F`,
RWTAC [ea_Xi_def, constT_inM_thm]);

val ea_Xrm_base_inM_thm = Q.prove (
`!iiid ro e. inM e (ea_Xrm_base iiid ro) ==> (?r. (ro = SOME r) /\ inM e (read_reg iiid r))`,
Cases_on `ro` THEN
RWTAC [ea_Xrm_base_def, constT_inM_thm]);

val ea_Xrm_index_inM_thm = Q.prove (
`!iiid ro e. 
  inM e (ea_Xrm_index iiid ro) 
  ==> 
  (?s r. (ro = SOME (s,r)) /\ inM e (read_reg iiid r))`,
Cases_on `ro` THEN
RWTAC [ea_Xrm_index_def, constT_inM_thm] THEN
Cases_on `x` THEN
FSTAC [ea_Xrm_index_def] THEN
IMP_RES_TAC seqT_inM_thm THEN
METIS_TAC [constT_inM_thm]);

val ea_Xrm_inM_thm = Q.prove (
`!ii X e. 
  inM e (ea_Xrm ii X) 
  ==> 
  (?i b d. (X = Xm i b d) /\ 
           (inM e (ea_Xrm_index ii i) \/ inM e (ea_Xrm_base ii b)))`,
Cases_on `X` THEN
RWTAC [ea_Xrm_def, ea_Xr_inM_thm] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC parT_inM_thm THEN
     RWTAC [ea_Xrm_index_inM_thm, ea_Xrm_base_inM_thm],
 Cases_on `x` THEN
     FSTAC [constT_inM_thm]]);

val ea_Xdest_inM_thm = Q.prove (
`!ii X e. 
  inM e (ea_Xdest ii X) 
  ==> 
  (?rm r. (X = Xrm_r rm r) /\ inM e (ea_Xrm ii rm)) \/
  (?rm i. (X = Xrm_i rm i) /\ inM e (ea_Xrm ii rm))`,
Cases_on `X` THEN
RWTAC [ea_Xdest_def, ea_Xr_inM_thm]);

val ea_Xsrc_inM_thm = Q.prove (
`!ii X e. 
  inM e (ea_Xsrc ii X) 
  ==> 
  ?r rm. (X = Xr_rm r rm) /\ inM e (ea_Xrm ii rm)`,
Cases_on `X` THEN
RWTAC [ea_Xsrc_def, ea_Xi_inM_thm, ea_Xr_inM_thm]);

val ea_Ximm_rm_inM_thm = Q.prove (
`!ii imm_rm e. 
  inM e (ea_Ximm_rm ii imm_rm) 
  ==> 
  ?rm. (imm_rm = Xi_rm rm) /\ inM e (ea_Xrm ii rm)`,
Cases_on `imm_rm` THEN
RWTAC [ea_Ximm_rm_def, ea_Xi_inM_thm]);

val read_ea_inM_thm = Q.prove (
`!ii ea e. 
  inM e (read_ea ii ea) 
  ==> 
  (?r. (ea = Xea_r r) /\ inM e (read_reg ii r)) \/
  (?a. (ea = Xea_m a) /\ inM e (read_m32 ii a))`,
Cases_on `ea` THEN
RWTAC [read_ea_def, constT_inM_thm]);

val read_src_ea_inM_thm = Q.prove (
`!ii ds e. 
  inM e (read_src_ea ii ds) 
  ==>
  inM e (ea_Xsrc ii ds) \/
  ?ea. inM e (read_ea ii ea)`,
RWTAC [read_src_ea_def] THEN
IMP_RES_TAC seqT_inM_thm THEN
METIS_TAC [addT_inM_thm]);

val read_dest_ea_inM_thm = Q.prove (
`!ii ds e. 
  inM e (read_dest_ea ii ds) 
  ==>
  inM e (ea_Xdest ii ds) \/
  ?ea. inM e (read_ea ii ea)`,
RWTAC [read_dest_ea_def] THEN
IMP_RES_TAC seqT_inM_thm THEN
METIS_TAC [addT_inM_thm]);

val write_ea_inM_thm = Q.prove (
`!ii ea x e. 
  inM e (write_ea ii ea x) 
  ==> 
  (?r. (ea = Xea_r r) /\ inM e (write_reg ii r x)) \/
  (?a. (ea = Xea_m a) /\ inM e (write_m32 ii a x))`,
Cases_on `ea` THEN
RWTAC [write_ea_def, constT_inM_thm, failureT_inM_thm]);

val write_logical_eflags_inM_thm = Q.prove (
`!ii w e. 
  inM e (write_logical_eflags ii w)
  ==>
  inM e (write_eflag ii X_AF NONE) \/
  inM e (write_eflag ii X_OF (SOME F)) \/
  inM e (write_eflag ii X_CF (SOME F)) \/
  (?v. inM e (write_eflag ii X_PF (SOME v))) \/
  (?v. inM e (write_eflag ii X_ZF (SOME v))) \/
  (?v. inM e (write_eflag ii X_SF (SOME v)))`,
RWTAC [write_logical_eflags_def, write_ZF_def, write_PF_def, write_SF_def] THEN
METIS_TAC [parT_unit_inM_thm]);

val write_arith_eflags_except_CF_inM_thm = Q.prove (
`!ii w e.
  inM e (write_arith_eflags_except_CF ii w) 
  ==>
  inM e (write_eflag ii X_OF NONE) \/
  inM e (write_eflag ii X_AF NONE) \/
  (?v. inM e (write_eflag ii X_PF (SOME v))) \/
  (?v. inM e (write_eflag ii X_ZF (SOME v))) \/
  (?v. inM e (write_eflag ii X_SF (SOME v)))`,
RWTAC [write_arith_eflags_except_CF_def, write_ZF_def, write_PF_def, write_SF_def] THEN
METIS_TAC [parT_unit_inM_thm]);

val erase_eflags_inM_thm = Q.prove (
`!ii e.
  inM e (erase_eflags ii) 
  ==>
  inM e (write_eflag ii X_OF NONE) \/
  inM e (write_eflag ii X_CF NONE) \/
  inM e (write_eflag ii X_AF NONE) \/
  inM e (write_eflag ii X_PF NONE) \/
  inM e (write_eflag ii X_ZF NONE) \/
  inM e (write_eflag ii X_SF NONE)`,
RWTAC [erase_eflags_def] THEN
METIS_TAC [parT_unit_inM_thm]);

val write_arith_eflags_inM_thm = Q.prove (
`!ii w c e.
  inM e (write_arith_eflags ii (w, c)) 
  ==>
  inM e (write_eflag ii X_CF (SOME c)) \/
  inM e (write_arith_eflags_except_CF ii w)`,
RWTAC [write_arith_eflags_def] THEN
METIS_TAC [parT_unit_inM_thm]);

val write_binop_Xcmp_inM_thm = Q.prove (
`!ii x y ea e.
  inM e (write_binop ii Xcmp x y ea)
  ==>
  (?w:bool[32] c. inM e (write_arith_eflags ii (w, c)))`,
RWTAC [write_binop_def, write_arith_result_no_write_def,
       sub_with_borrow_out_def] THEN
METIS_TAC []);

val write_result_erase_eflags_inM_thm = Q.prove (
`!ii e w.
  inM e (write_result_erase_eflags ii w ea)
  ==>
  inM e (erase_eflags ii) \/
  inM e (write_ea ii ea w)`,
RWTAC [write_result_erase_eflags_def] THEN
IMP_RES_TAC parT_unit_inM_thm THEN
RWTAC [] THEN
METIS_TAC [])

val write_binop_inM_thm = Q.prove (
`!ii X x y ea e.
  inM e (write_binop ii X x y ea)
  ==>
  (?w:bool[32] c. inM e (write_arith_eflags ii (w, c))) \/
  (?w. inM e (write_ea ii ea w)) \/
  (?w:bool[32]. inM e (write_logical_eflags ii w)) \/
  inM e (erase_eflags ii)`,
Cases_on `X` THEN
RWTAC [write_binop_def, write_arith_result_def, write_arith_result_no_write_def,
       write_logical_result_def, write_logical_result_no_write_def, 
       add_with_carry_out_def, sub_with_borrow_out_def,
        write_result_erase_eflags_def] THEN
IMP_RES_TAC parT_unit_inM_thm THEN
RWTAC [] THEN
METIS_TAC []);

val write_monop_inM_thm = Q.prove (
`!ii X x ea e.
  inM e (write_monop ii X x ea)
  ==>
  (?w:bool[32]. inM e (write_arith_eflags_except_CF ii w)) \/
  (?w. inM e (write_ea ii ea w)) \/
  inM e (write_eflag ii X_CF NONE)`,
Cases_on `X` THEN
RWTAC [write_monop_def, write_arith_result_no_CF_def] THEN
IMP_RES_TAC parT_unit_inM_thm THEN
IMP_RES_TAC parT_unit_inM_thm THEN
RWTAC [] THEN
METIS_TAC []);

val read_cond_inM_thm = Q.prove (
`!ii X e. inM e (read_cond ii X) ==> inM e (read_eflag ii X_ZF)`,
Cases_on `X` THEN
RWTAC [read_cond_def] THEN
IMP_RES_TAC seqT_inM_thm THEN
METIS_TAC [constT_inM_thm]);

val x86_exec_pop_inM_thm = Q.prove (
`!ii rm e.
  inM e (x86_exec_pop ii rm)
  ==>
  inM e (read_reg ii ESP) \/
  (?v. inM e (write_reg ii ESP v)) \/
  inM e (ea_Xrm ii rm) \/
  (?r. inM e (read_m32 ii r)) \/
  (?ea w. inM e (write_ea ii ea w))`,
RWTAC [x86_exec_pop_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC seqT_inM_thm THEN
     METIS_TAC [parT_inM_thm, addT_inM_thm],
 Cases_on `x` THEN
     FSTAC [] THEN
     IMP_RES_TAC seqT_inM_thm THENL
     [METIS_TAC [parT_inM_thm],
      METIS_TAC [parT_inM_thm],
      Cases_on `x` THEN
          FSTAC [] THEN
          METIS_TAC [], 
      Cases_on `x` THEN
          FSTAC [] THEN
          METIS_TAC []]]);

val x86_exec_push_inM_thm = Q.prove (
`!ii imm_rm e.
  inM e (x86_exec_push ii imm_rm)
  ==>
  inM e (ea_Ximm_rm ii imm_rm) \/
  (?ea. inM e (read_ea ii ea)) \/
  inM e (read_reg ii ESP) \/
  (?esp w. inM e (write_m32 ii esp w)) \/
  (?esp. inM e (write_reg ii ESP esp))`,
RWTAC [x86_exec_push_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC parT_inM_thm THENL
     [METIS_TAC [seqT_inM_thm],
      IMP_RES_TAC seqT_inM_thm THEN
          METIS_TAC [constT_inM_thm]],
 Cases_on `x` THEN
     FSTAC [] THEN
     METIS_TAC [parT_unit_inM_thm]]);

fun TRY_MP [] thm2 = MP_TAC thm2
  | TRY_MP (thm1::thms) thm2 =
      if can (MATCH_MP thm1) thm2 then
        STRIP_ASSUME_TAC (SIMP_RULE std_ss [] (MATCH_MP thm1 thm2))
      else 
        TRY_MP thms thm2;

val x86_exec_pop_eip_inM_thm = Q.prove (
`!ii e.
  inM e (x86_exec_pop_eip ii)
  ==>
  inM e (read_reg ii ESP) \/
  (?v. inM e (write_reg ii ESP v)) \/
  (?r. inM e (read_m32 ii r)) \/
  (?w. inM e (write_eip ii w))`,
RWTAC [x86_exec_pop_eip_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC seqT_inM_thm THEN
     METIS_TAC [parT_inM_thm, addT_inM_thm],
 Cases_on `x` THEN
     FSTAC [] THEN
     IMP_RES_TAC seqT_inM_thm THEN
     METIS_TAC []]);

val x86_exec_push_eip_inM_thm = Q.prove (
`!ii e.
  inM e (x86_exec_push_eip ii)
  ==>
  inM e (read_eip ii) \/
  inM e (read_reg ii ESP) \/
  (?esp w. inM e (write_m32 ii esp w)) \/
  (?esp. inM e (write_reg ii ESP esp))`,
RWTAC [x86_exec_push_eip_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC parT_inM_thm THENL
     [METIS_TAC [],
      IMP_RES_TAC seqT_inM_thm THEN
          METIS_TAC [constT_inM_thm]],
 Cases_on `x` THEN
     FSTAC [] THEN
     METIS_TAC [parT_unit_inM_thm]]);

val jump_to_ea_inM_thm = Q.prove (
`!ii eip X e.
  inM e (jump_to_ea ii eip X)
  ==>
  (?i. inM e (write_eip ii i)) \/
  (?r. inM e (read_reg ii r)) \/
  (?a. inM e (read_m32 ii a))`,
Cases_on `X` THEN
RWTAC [jump_to_ea_def] THEN
IMP_RES_TAC seqT_inM_thm THEN
METIS_TAC []);

val call_dest_from_ea_thm = Q.prove (
`!ii eip Xea e.
  inM e (call_dest_from_ea ii eip Xea)
  ==>
  (?r. inM e (read_reg ii r)) \/
  (?a. inM e (read_m32 ii a))`,
Cases_on `Xea` THEN
RWTAC [call_dest_from_ea_def] THEN
METIS_TAC [constT_inM_thm]);

fun TRY_MP [] thm2 = MP_TAC thm2
  | TRY_MP (thm1::thms) thm2 =
      if can (MATCH_MP thm1) thm2 then
        STRIP_ASSUME_TAC (SIMP_RULE std_ss [] (MATCH_MP thm1 thm2))
      else 
        TRY_MP thms thm2;

val inm_imp_thms = [if_inM_thm, mapT_inM_thm, constT_inM_thm, choiceT_inM_thm,
addT_inM_thm, failureT_inM_thm, seqT_inM_thm, parT_inM_thm, parT_unit_inM_thm,
write_reg_inM_thm, read_reg_inM_thm, write_eip_inM_thm, read_eip_inM_thm,
write_eflag_inM_thm, read_eflag_inM_thm, write_m32_inM_thm, read_m32_inM_thm,
ea_Xr_inM_thm, ea_Xi_inM_thm, ea_Xrm_base_inM_thm, ea_Xrm_index_inM_thm,
ea_Xrm_inM_thm, ea_Xdest_inM_thm, ea_Xsrc_inM_thm, ea_Ximm_rm_inM_thm,
read_ea_inM_thm, read_src_ea_inM_thm, read_dest_ea_inM_thm, write_ea_inM_thm,
write_logical_eflags_inM_thm, write_arith_eflags_except_CF_inM_thm,
write_arith_eflags_inM_thm, write_binop_Xcmp_inM_thm, write_binop_inM_thm,
write_monop_inM_thm, read_cond_inM_thm, x86_exec_pop_inM_thm,
x86_exec_push_inM_thm, x86_exec_pop_eip_inM_thm, x86_exec_push_eip_inM_thm,
jump_to_ea_inM_thm, call_dest_from_ea_thm, erase_eflags_inM_thm, 
write_result_erase_eflags_inM_thm];

val DO_INM =
  REPEAT (PAT_ASSUM ``inM x y`` (TRY_MP inm_imp_thms));

val existsM_def = Define `
  existsM P m = !eiid_start eiid_next x E. (eiid_next, x, E) IN m eiid_start ==> ?e. e IN E.events /\ P e`;

val if_existsM_thm = Q.prove (
`!b m m' P.
  (b /\ existsM e m) \/
  (~b /\ existsM e m')
  ==>
  existsM e (if b then m else m')`,
RWTAC []);

val mapT_existsM_thm = Q.prove (
`!f m P. existsM P m ==> existsM P (mapT_ev f m)`,
RWTAC [existsM_def, mapT_ev_def, LET_THM] THEN
METIS_TAC []);

val choiceT_existsM_thm = Q.prove (
`!m m' P. existsM P m /\ existsM P m' ==> existsM P (choiceT_ev m m')`,
RWTAC [choiceT_ev_def, existsM_def] THEN
METIS_TAC []);

val constT_existsM_thm = Q.prove (
`!x P. F ==> existsM P (constT x)`,
RWTAC []);

val addT_existsM_thm = Q.prove (
`!x s P. existsM P s ==> existsM P (addT x s)`,
RWTAC [addT_def, addT_ev_def, existsM_def, LET_THM] THEN
Cases_on `x''` THEN
Cases_on `r` THEN
FSTAC [] THEN
RWTAC [] THEN
METIS_TAC []);

val failureT_existsM_thm = Q.prove (
`!P. F ==> existsM e failureT`,
RWTAC []);

val seqT_existsM_thm = Q.prove (
`!m f P. 
  existsM P m \/ 
  (!x. (?eiid_next E eiid_start. (eiid_next, x, E) IN m eiid_start) ==> existsM P (f x))
  ==> 
  existsM P (seqT m f)`,
RWTAC [existsM_def, seqT_def, seqT_ev_def, LET_THM, event_structure_seq_union_def] THEN
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN
METIS_TAC []);

val parT_existsM_thm = Q.prove (
`!m m' P. existsM P m \/ existsM P m' ==> existsM P (parT m m')`,
RWTAC [existsM_def, parT_def, parT_ev_def, LET_THM, event_structure_union_def] THEN
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN
METIS_TAC []);

(* Cut and pasted from above *)
val parT_unit_existsM_thm = Q.prove (
`!m m' P. existsM P m \/ existsM P m' ==> existsM P (parT_unit m m')`,
RWTAC [existsM_def, parT_unit_def, parT_unit_ev_def, LET_THM, event_structure_union_def] THEN
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN
METIS_TAC []);

(*
val write_reg_inM_thm = Q.prove (
`!ii r x e.
  inM e (write_reg ii r x) 
  ==> 
  (loc e = SOME (Location_reg ii.proc (Reg32 r))) /\
  (e.action = Access W (Location_reg ii.proc (Reg32 r)) x)`,
RWTAC [inM_def, write_reg_def, write_reg_ev_def, write_location_ev_def,
       loc_def] THEN
FSTAC []);

val read_reg_inM_thm = Q.prove (
`!ii r e.
  inM e (read_reg ii r) 
  ==> 
  (loc e = SOME (Location_reg ii.proc (Reg32 r))) /\
  (?v. e.action = Access R (Location_reg ii.proc (Reg32 r)) v)`,
RWTAC [inM_def, read_reg_def, read_reg_ev_def, read_location_ev_def,
       loc_def] THEN
FSTAC []);

val write_eip_inM_thm = Q.prove (
`!ii x e.
  inM e (write_eip ii x) 
  ==> 
  (loc e = SOME (Location_reg ii.proc RegEIP)) /\
  (e.action = Access W (Location_reg ii.proc RegEIP) x)`,
RWTAC [inM_def, write_eip_def, write_eip_ev_def, write_location_ev_def,
       loc_def] THEN
FSTAC []);

val read_eip_inM_thm = Q.prove (
`!ii e.
  inM e (read_eip ii) 
  ==> 
  (loc e = SOME (Location_reg ii.proc RegEIP)) /\
  (?v. e.action = Access R (Location_reg ii.proc RegEIP) v)`,
RWTAC [inM_def, read_eip_def, read_eip_ev_def, read_location_ev_def,
       loc_def] THEN
FSTAC []);

val write_eflag_inM_thm = Q.prove (
`!ii f bo e.
  inM e (write_eflag ii f bo)
  ==>
  (loc e = SOME (Location_reg ii.proc (Reg1 f))) /\
  (?v. e.action = Access W (Location_reg ii.proc (Reg1 f)) v)`,
Cases_on `bo` THEN
RWTAC [write_eflag_def, write_eflag_ev_def, write_location_ev_def, loc_def] THEN
IMP_RES_TAC choiceT_inM_thm THEN
FSTAC [inM_def] THEN
RWTAC [] THEN 
FSTAC [] THEN
RWTAC []);

val read_eflag_inM_thm = Q.prove (
`!ii f e.
  inM e (read_eflag ii f) 
  ==>
  (loc e = SOME (Location_reg ii.proc (Reg1 f))) /\
  (?v. e.action = Access R (Location_reg ii.proc (Reg1 f)) v)`,
RWTAC [read_eflag_def, read_eflag_ev_def, read_location_ev_def, loc_def] THEN
IMP_RES_TAC mapT_inM_thm THEN
FSTAC [inM_def] THEN
RWTAC [] THEN 
FSTAC [] THEN
RWTAC []);

val write_m32_inM_thm = Q.prove (
`!ii a x e.
  inM e (write_m32 ii a x)
  ==>
  (loc e = SOME (Location_mem a)) /\
  (e.action = Access W (Location_mem a) x)`,
RWTAC [write_m32_def, write_m32_ev_def, write_location_ev_def,
inM_def, loc_def, failureT_ev_def] THEN
FSTAC []);

*)

val read_m32_existsM_thm = Q.prove (
`!ii a P.
  (!v eiid. P (<|eiid:=eiid; iiid:=ii; action:=Access R (Location_mem a) v|>))
  ==>
  existsM P (read_m32 ii a)`,
RWTAC [read_m32_def, read_m32_ev_def, read_location_ev_def,
       existsM_def, failureT_ev_def] THEN
FSTAC []);

val ea_Xr_existsM_thm = Q.prove (
`!r P. F ==> existsM P (ea_Xr r)`,
RWTAC [ea_Xr_def, constT_existsM_thm]);

val ea_Xi_existsM_thm = Q.prove (
`!i P. F ==> existsM P (ea_Xi i)`,
RWTAC [ea_Xi_def, constT_existsM_thm]);

val ea_Xrm_base_existsM_thm = Q.prove (
`!iiid ro P. (?r. (ro = SOME r) /\ existsM P (read_reg iiid r)) ==> existsM P (ea_Xrm_base iiid ro)`,
Cases_on `ro` THEN
RWTAC [ea_Xrm_base_def, constT_existsM_thm]);

val ea_Xrm_index_existsM_thm = Q.prove (
`!iiid ro P. 
  (?s r. (ro = SOME (s,r)) /\ existsM P (read_reg iiid r))
  ==> 
  existsM P (ea_Xrm_index iiid ro)`, 
Cases_on `ro` THEN
RWTAC [] THEN
FSTAC [ea_Xrm_index_def] THEN
METIS_TAC [seqT_existsM_thm]);

val ea_Xrm_existsM_thm = Q.prove (
`!ii X P. 
  (?i b d. (X = Xm i b d) /\ 
           (existsM P (ea_Xrm_index ii i) \/ existsM P (ea_Xrm_base ii b)))
  ==>
  existsM P (ea_Xrm ii X)`, 
Cases_on `X` THEN
RWTAC [ea_Xrm_def, ea_Xr_existsM_thm] THEN
MATCH_MP_TAC seqT_existsM_thm THEN
DISJ1_TAC THEN
MATCH_MP_TAC parT_existsM_thm THEN
METIS_TAC []);

val ea_Xdest_existsM_thm = Q.prove (
`!ii X P. 
  (?rm r. (X = Xrm_r rm r) /\ existsM P (ea_Xrm ii rm)) \/
  (?rm i. (X = Xrm_i rm i) /\ existsM P (ea_Xrm ii rm))
  ==>
  existsM P (ea_Xdest ii X)`, 
Cases_on `X` THEN
RWTAC [ea_Xdest_def, ea_Xr_existsM_thm]);

val ea_Xsrc_existsM_thm = Q.prove (
`!ii X P. 
  (?r rm. (X = Xr_rm r rm) /\ existsM P (ea_Xrm ii rm))
  ==>
  existsM P (ea_Xsrc ii X)`,
Cases_on `X` THEN
RWTAC [ea_Xsrc_def, ea_Xi_existsM_thm, ea_Xr_existsM_thm]);

(*
val ea_Ximm_rm_inM_thm = Q.prove (
`!ii imm_rm e. 
  inM e (ea_Ximm_rm ii imm_rm) 
  ==> 
  ?rm. (imm_rm = Xi_rm rm) /\ inM e (ea_Xrm ii rm)`,
Cases_on `imm_rm` THEN
RWTAC [ea_Ximm_rm_def, ea_Xi_inM_thm]);
*)

val read_ea_existsM_thm = Q.prove (
`!ii ea P. 
  (?r. (ea = Xea_r r) /\ existsM P (read_reg ii r)) \/
  (?a. (ea = Xea_m a) /\ existsM P (read_m32 ii a))
  ==>
  existsM P (read_ea ii ea)`, 
Cases_on `ea` THEN
RWTAC [read_ea_def]);

val read_src_ea_existsM_thm = Q.prove (
`!ii ds P. 
  existsM P (ea_Xsrc ii ds) \/
  (!ea. (?eiid_next E eiid_start. (eiid_next, ea, E) IN (ea_Xsrc ii ds) eiid_start) ==> existsM P (read_ea ii ea))
  ==>
  existsM P (read_src_ea ii ds)`,
RWTAC [read_src_ea_def] THEN
MATCH_MP_TAC seqT_existsM_thm THEN
RWTAC [] THEN 
METIS_TAC [addT_existsM_thm]);

val read_dest_ea_existsM_thm = Q.prove (
`!ii ds P. 
  existsM P (ea_Xdest ii ds) \/
  (!ea. (?eiid_next E eiid_start. (eiid_next, ea, E) IN (ea_Xdest ii ds) eiid_start) ==> existsM P (read_ea ii ea))
  ==>
  existsM P (read_dest_ea ii ds)`, 
RWTAC [read_dest_ea_def] THEN
MATCH_MP_TAC seqT_existsM_thm THEN
RWTAC [] THEN 
METIS_TAC [addT_existsM_thm]);

(*
val write_ea_inM_thm = Q.prove (
`!ii ea x e. 
  inM e (write_ea ii ea x) 
  ==> 
  (?r. (ea = Xea_r r) /\ inM e (write_reg ii r x)) \/
  (?a. (ea = Xea_m a) /\ inM e (write_m32 ii a x))`,
Cases_on `ea` THEN
RWTAC [write_ea_def, constT_inM_thm, failureT_inM_thm]);

val write_logical_eflags_inM_thm = Q.prove (
`!ii w e. 
  inM e (write_logical_eflags ii w)
  ==>
  inM e (write_eflag ii X_AF NONE) \/
  inM e (write_eflag ii X_OF (SOME F)) \/
  inM e (write_eflag ii X_CF (SOME F)) \/
  (?v. inM e (write_eflag ii X_PF (SOME v))) \/
  (?v. inM e (write_eflag ii X_ZF (SOME v))) \/
  (?v. inM e (write_eflag ii X_SF (SOME v)))`,
RWTAC [write_logical_eflags_def, write_ZF_def, write_PF_def, write_SF_def] THEN
METIS_TAC [parT_unit_inM_thm]);

val write_arith_eflags_except_CF_inM_thm = Q.prove (
`!ii w e.
  inM e (write_arith_eflags_except_CF ii w) 
  ==>
  inM e (write_eflag ii X_OF NONE) \/
  inM e (write_eflag ii X_AF NONE) \/
  (?v. inM e (write_eflag ii X_PF (SOME v))) \/
  (?v. inM e (write_eflag ii X_ZF (SOME v))) \/
  (?v. inM e (write_eflag ii X_SF (SOME v)))`,
RWTAC [write_arith_eflags_except_CF_def, write_ZF_def, write_PF_def, write_SF_def] THEN
METIS_TAC [parT_unit_inM_thm]);

val write_arith_eflags_inM_thm = Q.prove (
`!ii w c e.
  inM e (write_arith_eflags ii (w, c)) 
  ==>
  inM e (write_eflag ii X_CF (SOME c)) \/
  inM e (write_arith_eflags_except_CF ii w)`,
RWTAC [write_arith_eflags_def] THEN
METIS_TAC [parT_unit_inM_thm]);

val write_binop_Xcmp_inM_thm = Q.prove (
`!ii x y ea e.
  inM e (write_binop ii Xcmp x y ea)
  ==>
  (?w:bool[32] c. inM e (write_arith_eflags ii (w, c)))`,
RWTAC [write_binop_def, write_arith_result_no_write_def,
       sub_with_borrow_out_def] THEN
METIS_TAC []);

val write_binop_inM_thm = Q.prove (
`!ii X x y ea e.
  inM e (write_binop ii X x y ea)
  ==>
  (?w:bool[32] c. inM e (write_arith_eflags ii (w, c))) \/
  (?w. inM e (write_ea ii ea w)) \/
  (?w:bool[32]. inM e (write_logical_eflags ii w))`,
Cases_on `X` THEN
RWTAC [write_binop_def, write_arith_result_def, write_arith_result_no_write_def,
       write_logical_result_def, write_logical_result_no_write_def, 
       add_with_carry_out_def, sub_with_borrow_out_def] THEN
IMP_RES_TAC parT_unit_inM_thm THEN
RWTAC [] THEN
METIS_TAC []);

val write_monop_inM_thm = Q.prove (
`!ii X x ea e.
  inM e (write_monop ii X x ea)
  ==>
  (?w:bool[32]. inM e (write_arith_eflags_except_CF ii w)) \/
  (?w. inM e (write_ea ii ea w)) \/
  inM e (write_eflag ii X_CF NONE)`,
Cases_on `X` THEN
RWTAC [write_monop_def, write_arith_result_no_CF_def] THEN
IMP_RES_TAC parT_unit_inM_thm THEN
IMP_RES_TAC parT_unit_inM_thm THEN
RWTAC [] THEN
METIS_TAC []);

val read_cond_inM_thm = Q.prove (
`!ii X e. inM e (read_cond ii X) ==> inM e (read_eflag ii X_ZF)`,
Cases_on `X` THEN
RWTAC [read_cond_def] THEN
IMP_RES_TAC seqT_inM_thm THEN
METIS_TAC [constT_inM_thm]);

val x86_exec_pop_inM_thm = Q.prove (
`!ii rm e.
  inM e (x86_exec_pop ii rm)
  ==>
  inM e (read_reg ii ESP) \/
  (?v. inM e (write_reg ii ESP v)) \/
  inM e (ea_Xrm ii rm) \/
  (?r. inM e (read_m32 ii r)) \/
  (?ea w. inM e (write_ea ii ea w))`,
RWTAC [x86_exec_pop_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC seqT_inM_thm THEN
     METIS_TAC [parT_inM_thm, addT_inM_thm],
 Cases_on `x` THEN
     FSTAC [] THEN
     IMP_RES_TAC seqT_inM_thm THENL
     [METIS_TAC [parT_inM_thm],
      METIS_TAC [parT_inM_thm],
      Cases_on `x` THEN
          FSTAC [] THEN
          METIS_TAC [], 
      Cases_on `x` THEN
          FSTAC [] THEN
          METIS_TAC []]]);

val x86_exec_push_inM_thm = Q.prove (
`!ii imm_rm e.
  inM e (x86_exec_push ii imm_rm)
  ==>
  inM e (ea_Ximm_rm ii imm_rm) \/
  (?ea. inM e (read_ea ii ea)) \/
  inM e (read_reg ii ESP) \/
  (?esp w. inM e (write_m32 ii esp w)) \/
  (?esp. inM e (write_reg ii ESP esp))`,
RWTAC [x86_exec_push_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC parT_inM_thm THENL
     [METIS_TAC [seqT_inM_thm],
      IMP_RES_TAC seqT_inM_thm THEN
          METIS_TAC [constT_inM_thm]],
 Cases_on `x` THEN
     FSTAC [] THEN
     METIS_TAC [parT_unit_inM_thm]]);

fun TRY_MP [] thm2 = MP_TAC thm2
  | TRY_MP (thm1::thms) thm2 =
      if can (MATCH_MP thm1) thm2 then
        STRIP_ASSUME_TAC (SIMP_RULE std_ss [] (MATCH_MP thm1 thm2))
      else 
        TRY_MP thms thm2;

val x86_exec_pop_eip_inM_thm = Q.prove (
`!ii e.
  inM e (x86_exec_pop_eip ii)
  ==>
  inM e (read_reg ii ESP) \/
  (?v. inM e (write_reg ii ESP v)) \/
  (?r. inM e (read_m32 ii r)) \/
  (?w. inM e (write_eip ii w))`,
RWTAC [x86_exec_pop_eip_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC seqT_inM_thm THEN
     METIS_TAC [parT_inM_thm, addT_inM_thm],
 Cases_on `x` THEN
     FSTAC [] THEN
     IMP_RES_TAC seqT_inM_thm THEN
     METIS_TAC []]);

val x86_exec_push_eip_inM_thm = Q.prove (
`!ii e.
  inM e (x86_exec_push_eip ii)
  ==>
  inM e (read_eip ii) \/
  inM e (read_reg ii ESP) \/
  (?esp w. inM e (write_m32 ii esp w)) \/
  (?esp. inM e (write_reg ii ESP esp))`,
RWTAC [x86_exec_push_eip_def] THEN
IMP_RES_TAC seqT_inM_thm THENL
[IMP_RES_TAC parT_inM_thm THENL
     [METIS_TAC [],
      IMP_RES_TAC seqT_inM_thm THEN
          METIS_TAC [constT_inM_thm]],
 Cases_on `x` THEN
     FSTAC [] THEN
     METIS_TAC [parT_unit_inM_thm]]);
     *)

val lockT_existsM_thm = Q.prove (
`!P m. existsM P m ==> existsM P (lockT m)`,
RWTAC [lockT_def, lockT_ev_def, existsM_def, event_structure_lock_def, LET_THM] THEN
Cases_on `x'` THEN
Cases_on `r` THEN
FSTAC [] THEN
RWTAC [] THEN
METIS_TAC []);


val existsm_imp_thms = [if_existsM_thm, mapT_existsM_thm, constT_existsM_thm,
choiceT_existsM_thm, addT_existsM_thm, failureT_existsM_thm, seqT_existsM_thm,
parT_existsM_thm, parT_unit_existsM_thm, read_m32_existsM_thm,
ea_Xr_existsM_thm, ea_Xi_existsM_thm, ea_Xrm_base_existsM_thm,
ea_Xrm_index_existsM_thm, ea_Xrm_existsM_thm, ea_Xdest_existsM_thm,
ea_Xsrc_existsM_thm, read_ea_existsM_thm, read_src_ea_existsM_thm,
read_dest_ea_existsM_thm, lockT_existsM_thm];

val DO_EXISTSM = 
  DEPTH_CONSEQ_CONV_TAC (CONSEQ_REWRITE_CONV existsm_imp_thms) THEN
  REWRITE_TAC [GSYM DISJ_ASSOC];

val wf_def = Define `
  wf E = 
    FINITE E.events /\
    (E.procs = { e.iiid.proc | e IN E.events }) /\
    (!e :: (E.events). proc e IN E.procs) /\
    (!e1 e2 :: (E.events). (e1.eiid = e2.eiid) ==> (e1 = e2)) /\
    (DOM E.intra_causality) SUBSET E.events /\
    (RANGE E.intra_causality) SUBSET E.events /\
    acyclic (E.intra_causality) /\
    (!e1 e2 :: (E.events). (e1.iiid = e2.iiid)) /\
    (!(e1,e2)::(E.intra_causality). ~mem_store e1) /\
    (!(e1 :: writes E) e2.
      ~(e1 = e2) /\
      (e2 IN writes E \/ e2 IN reads E) /\
      (e1.iiid = e2.iiid) /\ 
      (loc e1 = loc e2) /\
      (?p r. loc e1 = SOME (Location_reg p r))
      ==>
      (e1, e2) IN sTC E.intra_causality \/
      (e2, e1) IN sTC E.intra_causality) /\
    (E.atomicity = {}) /\
    (!e :: (E.events). !p r. (loc e = SOME (Location_reg p r)) ==> (p = proc e))`;

val wf0 = Q.prove (
`!E. wf E ==> FINITE E.events`,
RWTAC [wf_def]);

val wf1 = Q.prove (
`!E. wf E ==> (!e1 e2 :: (E.events). (e1.eiid = e2.eiid) ==> (e1 = e2))`,
RWTAC [wf_def]);

val wf2 = Q.prove (
`!E. wf E ==> (DOM E.intra_causality) SUBSET E.events`,
RWTAC [wf_def]);

val wf3 = Q.prove (
`!E. wf E ==> (RANGE E.intra_causality) SUBSET E.events`,
RWTAC [wf_def]);

val wf4 = Q.prove (
`!E. wf E ==> acyclic (E.intra_causality)`,
RWTAC [wf_def]);

val wf5 = Q.prove (
`!E. wf E ==> (!e1 e2 :: (E.events). (e1.iiid = e2.iiid))`,
RWTAC [wf_def]);

val wf6 = Q.prove (
`!E. wf E ==>
  (!(e1 :: writes E) e2.
    ~(e1 = e2) /\
    (e2 IN writes E \/ e2 IN reads E) /\
    (e1.iiid = e2.iiid) /\ 
    (loc e1 = loc e2) /\
    (?p r. loc e1 = SOME (Location_reg p r))
    ==>
    (e1, e2) IN sTC E.intra_causality \/
    (e2, e1) IN sTC E.intra_causality)`,
RWTAC [wf_def]);

val wf7 = Q.prove (
`!E. wf E ==> (E.atomicity = {})`,
RWTAC [wf_def]);

val wf8 = Q.prove (
`!E. wf E ==> (!e :: (E.events). !p r. (loc e = SOME (Location_reg p r)) ==> (p = proc e))`,
RWTAC [wf_def]);

val wf9 = Q.prove (
`!E. wf E ==> (!(e1,e2)::(E.intra_causality). ~mem_store e1)`,
RWTAC [wf_def]);

val wf10 = Q.prove (
`!E. wf E ==> (E.procs = { e.iiid.proc | e IN E.events })`,
RWTAC [wf_def]);

val wf_thm = Q.prove (
`!E. wf E ==> well_formed_event_structure E`,
RWTAC [wf_def, well_formed_event_structure_def] THENL
[`{eiid | ?e. e IN E.events /\ (e.iiid = iiid) /\ (e.eiid = eiid)} SUBSET (IMAGE (\e. e.eiid) E.events)` by
             (RWTAC [SUBSET_DEF, IMAGE_DEF] THEN
              METIS_TAC []) THEN
     METIS_TAC [SUBSET_FINITE, IMAGE_FINITE],
 METIS_TAC [IMAGE_FINITE, IMAGE_DEF],
 FSTAC [DOM_def, RANGE_def, SUBSET_DEF] THEN
     METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 RWTAC [PER_def]]);

val wfM_def = Define `
  wfM id m =
    !eiid_start eiid_next x E. 
      (eiid_next, x, E) IN m eiid_start 
      ==> 
      wf E /\
      (!e. e IN E.events ==> (e.iiid = id)) /\
      (!e. e IN E.events ==> ~(e.eiid IN eiid_start)) /\
      (eiid_next = eiid_start UNION {e.eiid | e IN E.events})`;

val if_wf_thm = Q.prove (
`!ii b m m'.
  (b ==> wfM ii m) /\
  (~b ==> wfM ii m')
  ==>
  wfM ii (if b then m else m')`,
RWTAC []);

val es_union_thm = Q.prove (
`!E E'. 
  wf E /\
  wf E' /\
  (!e e'. e IN E.events /\ e' IN E'.events ==> (e.iiid = e'.iiid)) /\
  (!e e'. e IN E.events /\ e' IN E'.events ==> ~(e.eiid = e'.eiid)) /\
  (!e e'. e IN E.events /\ e' IN E'.events /\ (loc e = loc e') /\
          (?p r. loc e = SOME (Location_reg p r)) ==>
          (?l v. e.action = Access R l v) /\
          (?l v. e'.action = Access R l v))
  ==>
  wf (event_structure_union E E')`,
RWTAC [] THEN
RWTAC [wf_def, event_structure_union_def] THENL
[METIS_TAC [RQ wf0],
 METIS_TAC [RQ wf0],
 RWTAC [EXTENSION] THEN
     METIS_TAC [RQ (SIMP_RULE (srw_ss()) [EXTENSION] wf10)],
 FSTAC [proc_def] THEN
     METIS_TAC [RQ (SIMP_RULE (srw_ss()) [EXTENSION] wf10)],
 FSTAC [proc_def] THEN
     METIS_TAC [RQ (SIMP_RULE (srw_ss()) [EXTENSION] wf10)],
 METIS_TAC [RQ wf1],
 METIS_TAC [RQ wf1],
 METIS_TAC [RQ wf1],
 METIS_TAC [RQ wf1],
 IMP_RES_TAC wf2 THEN
     FSTAC [DOM_def, SUBSET_DEF, minimal_elements_def, maximal_elements_def] THEN
     METIS_TAC [],
 IMP_RES_TAC wf3 THEN
     FSTAC [RANGE_def, SUBSET_DEF, minimal_elements_def, maximal_elements_def] THEN
     METIS_TAC [],
 MATCH_MP_TAC acyclic_union_lem THEN
     FSTAC [wf_def, DISJOINT_DEF, EXTENSION, DOM_def, RANGE_def, SUBSET_DEF] THEN
     METIS_TAC [],
 METIS_TAC [RQ wf5],
 METIS_TAC [RQ wf5],
 METIS_TAC [RQ wf5],
 METIS_TAC [RQ wf5],
 METIS_TAC [RQ wf9],
 METIS_TAC [RQ wf9],
 IMP_RES_TAC wf6 THEN
     FSTAC [writes_def] THENL
     [METIS_TAC [sTC_UNION_lem],
      FSTAC [writes_def, reads_def] THEN
          METIS_TAC [action_11, dirn_distinct],
      FSTAC [writes_def, reads_def] THEN
          METIS_TAC [action_11, dirn_distinct],
      METIS_TAC [sTC_UNION_lem, UNION_COMM]],
 IMP_RES_TAC wf6 THEN
     FSTAC [writes_def, reads_def] THENL
     [METIS_TAC [sTC_UNION_lem],
      FSTAC [writes_def, reads_def] THEN
          METIS_TAC [action_11, dirn_distinct],
      FSTAC [writes_def, reads_def] THEN
          METIS_TAC [action_11, dirn_distinct],
      METIS_TAC [sTC_UNION_lem, UNION_COMM]],
 METIS_TAC [RQ wf7],
 METIS_TAC [RQ wf7],
 METIS_TAC [RQ wf8],
 METIS_TAC [RQ wf8]]);

val lem = Q.prove (
`!e e' E E' f g. 
  e IN f E.events E.intra_causality /\ 
  e' IN g E'.events E'.intra_causality 
  ==> 
  (e, e') IN {(e1, e2) | e1 IN f E.events E.intra_causality /\ e2 IN  g E'.events E'.intra_causality}`,
RWTAC []);

val T = 
IMP_RES_TAC wf6 THEN
FSTAC [writes_def, reads_def] THENL
[METIS_TAC [sTC_UNION_lem],
 `e1 IN maximal_elements E.events E.intra_causality \/ 
  ?e. e IN maximal_elements E.events E.intra_causality /\ (e1, e) IN sTC E.intra_causality` 
               by METIS_TAC [finite_acyclic_has_maximal_path_thm, wf0, wf4] THEN
     `e2 IN minimal_elements E'.events E'.intra_causality \/ 
      ?e. (e, e2) IN sTC E'.intra_causality /\ e IN minimal_elements E'.events E'.intra_causality` 
               by METIS_TAC [finite_acyclic_has_minimal_path_thm, wf0, wf4] THEN
     IMP_RES_TAC lem THEN
     Q.ABBREV_TAC `new = {(e1,e2) | e1 IN maximal_elements E.events E.intra_causality /\ 
                                    e2 IN minimal_elements E'.events E'.intra_causality}` THENL
     [METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules],
      `(e1, e) IN sTC (E.intra_causality UNION E'.intra_causality UNION new)`
               by METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules] THEN
          METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules],
      `(e, e2) IN sTC (E.intra_causality UNION E'.intra_causality UNION new)`
               by METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules] THEN
          METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules],
      `(e, e') IN sTC (E.intra_causality UNION E'.intra_causality UNION new)`
               by METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules] THEN
          METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules]],
 `e2 IN maximal_elements E.events E.intra_causality \/ 
  ?e. e IN maximal_elements E.events E.intra_causality /\ (e2, e) IN sTC E.intra_causality` 
               by METIS_TAC [finite_acyclic_has_maximal_path_thm, wf0, wf4] THEN
     `e1 IN minimal_elements E'.events E'.intra_causality \/ 
      ?e. (e, e1) IN sTC E'.intra_causality /\ e IN minimal_elements E'.events E'.intra_causality` 
               by METIS_TAC [finite_acyclic_has_minimal_path_thm, wf0, wf4] THEN
     IMP_RES_TAC lem THEN
     Q.ABBREV_TAC `new = {(e1,e2) | e1 IN maximal_elements E.events E.intra_causality /\ 
                                    e2 IN minimal_elements E'.events E'.intra_causality}` THENL
     [METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules],
      `(e2, e) IN sTC (E.intra_causality UNION E'.intra_causality UNION new)`
               by METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules] THEN
          METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules],
      `(e, e1) IN sTC (E.intra_causality UNION E'.intra_causality UNION new)`
               by METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules] THEN
          METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules],
      `(e, e') IN sTC (E.intra_causality UNION E'.intra_causality UNION new)`
               by METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules] THEN
          METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules]],
 METIS_TAC [sTC_UNION_lem, UNION_COMM]];

val es_seq_union_thm = Q.prove (
`!E E'. 
  wf E /\
  wf E' /\
  (!e e'. e IN E.events /\ e' IN E'.events ==> (e.iiid = e'.iiid)) /\
  (!e e'. e IN E.events /\ e' IN E'.events ==> ~(e.eiid = e'.eiid)) /\
  (!e. e IN E.events ==> ~mem_store e)
  ==>
  wf (event_structure_seq_union E E')`,
RWTAC [] THEN
RWTAC [wf_def, event_structure_seq_union_def] THENL
[METIS_TAC [wf0],
 METIS_TAC [wf0],
 RWTAC [EXTENSION] THEN
     METIS_TAC [RQ (SIMP_RULE (srw_ss()) [EXTENSION] wf10)],
 FSTAC [proc_def] THEN
     METIS_TAC [RQ (SIMP_RULE (srw_ss()) [EXTENSION] wf10)],
 FSTAC [proc_def] THEN
     METIS_TAC [RQ (SIMP_RULE (srw_ss()) [EXTENSION] wf10)],
 METIS_TAC [wf1],
 METIS_TAC [wf1],
 METIS_TAC [wf1],
 METIS_TAC [wf1],
 IMP_RES_TAC wf2 THEN
     FSTAC [DOM_def, SUBSET_DEF, minimal_elements_def, maximal_elements_def] THEN
     METIS_TAC [],
 IMP_RES_TAC wf3 THEN
     FSTAC [RANGE_def, SUBSET_DEF, minimal_elements_def, maximal_elements_def] THEN
     METIS_TAC [],
 MATCH_MP_TAC acyclic_seq_union_lem THEN
     FSTAC [wf_def, SUBSET_DEF, DOM_def, RANGE_def, DISJOINT_DEF,
            EXTENSION] THEN
     METIS_TAC [],
 METIS_TAC [RQ wf5],
 METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [RQ wf5],
 METIS_TAC [RQ wf9],
 METIS_TAC [RQ wf9],
 FSTAC [maximal_elements_def],
 T,
 T,
 METIS_TAC [RQ wf7],
 METIS_TAC [RQ wf7],
 METIS_TAC [RQ wf8],
 METIS_TAC [RQ wf8]]);

val mapT_wf_thm = Q.prove (
`!ii f m. wfM ii m ==> wfM ii (mapT_ev f m)`,
RWTAC [mapT_ev_def, wfM_def, LET_THM] THEN
METIS_TAC []);

val choiceT_wf_thm = Q.prove (
`!ii m m'. wfM ii m /\ wfM ii m' ==> wfM ii (choiceT_ev m m')`,
RWTAC [wfM_def, choiceT_ev_def] THEN
METIS_TAC []);

val constT_wf_thm = Q.prove (
`!ii x. wfM ii (constT x)`,
RWTAC [wfM_def, constT_def, constT_ev_def, wf_def, event_structure_empty_def,
       DOM_def, RANGE_def, acyclic_def, writes_def, sTC_EMPTY, EXTENSION]);

val addT_wf_thm = Q.prove (
`!ii x s. wfM ii s ==> wfM ii (addT x s)`,
RWTAC [addT_def, addT_ev_def, wfM_def, LET_THM] THEN
Cases_on `x''` THEN
Cases_on `r` THEN
FSTAC [] THEN
RWTAC [] THEN
RES_TAC);

val failureT_wf_thm = Q.prove (
`!ii. wfM ii failureT`,
RWTAC [wfM_def, failureT_def, failureT_ev_def]);

val seqT_wf_ctxt_thm = Q.prove (
`!iiid m f. 
  wfM iiid m /\
  (!x. (?eiid_next E eiid_start. (eiid_next, x, E) IN m eiid_start) ==> wfM iiid (f x)) /\
  (!e. inM e m ==> ~mem_store e)
  ==>
  wfM iiid (seqT m f)`,
RWTAC [wfM_def, seqT_def, seqT_ev_def, LET_THM] THEN
FSTAC [] THEN
RWTAC [] THENL
[RES_TAC THEN
    MATCH_MP_TAC es_seq_union_thm THEN
    RWTAC [] THEN
    RES_TAC THEN
    FSTAC [] THEN
    METIS_TAC [inM_def],
 FSTAC [event_structure_seq_union_def] THEN
     METIS_TAC [],
 FSTAC [event_structure_seq_union_def] THEN
     RES_TAC THEN
     FSTAC [],
 FSTAC [event_structure_seq_union_def, EXTENSION] THEN
     RWTAC [] THEN
     EQ_TAC THEN
     RWTAC [] THEN
     RES_TAC THEN 
     RWTAC [] THEN 
     METIS_TAC []]);

val parT_wf_thm = Q.prove (
`!iiid m m'. 
  wfM iiid m /\
  wfM iiid m' /\
  (!e e'.
    (loc e = loc e') /\
    (?p r. loc e = SOME (Location_reg p r)) /\
    inM e m /\
    inM e' m'
    ==>
    (?l v. e.action = Access R l v) /\
    (?l v. e'.action = Access R l v)) 
  ==>
  wfM iiid (parT m m')`,
RWTAC [inM_def, wfM_def, parT_def, parT_ev_def, LET_THM] THEN
FSTAC [] THEN
RWTAC [] THENL
[RES_TAC THEN
    MATCH_MP_TAC es_union_thm THEN
    RWTAC [] THENL
    [RES_TAC THEN
         FSTAC [] THEN
         METIS_TAC [],
     METIS_TAC [],
     METIS_TAC []],
 FSTAC [event_structure_union_def] THEN
     METIS_TAC [],
 FSTAC [event_structure_union_def] THEN
     RES_TAC THEN
     FSTAC [],
 FSTAC [event_structure_union_def, EXTENSION] THEN
     RWTAC [] THEN
     EQ_TAC THEN
     RWTAC [] THENL
     [RES_TAC THEN RWTAC [] THEN METIS_TAC [],
      METIS_TAC [],
      METIS_TAC [],
      METIS_TAC []]]);

(* Cut and pasted from above *)
val parT_unit_wf_thm = Q.prove (
`!iiid m m'. 
  wfM iiid m /\
  wfM iiid m' /\
  (!e e'.
    (loc e = loc e') /\
    (?p r. loc e = SOME (Location_reg p r)) /\
    inM e m /\
    inM e' m'
    ==>
    (?l v. e.action = Access R l v) /\
    (?l v. e'.action = Access R l v)) 
  ==>
  wfM iiid (parT_unit m m')`,
RWTAC [inM_def, wfM_def, parT_unit_def, parT_unit_ev_def, LET_THM] THEN
FSTAC [] THEN
RWTAC [] THENL
[RES_TAC THEN
    MATCH_MP_TAC es_union_thm THEN
    RWTAC [] THENL
    [RES_TAC THEN
         FSTAC [] THEN
         METIS_TAC [],
     METIS_TAC [],
     METIS_TAC []],
 FSTAC [event_structure_union_def] THEN
     METIS_TAC [],
 FSTAC [event_structure_union_def] THEN
     RES_TAC THEN
     FSTAC [],
 FSTAC [event_structure_union_def, EXTENSION] THEN
     RWTAC [] THEN
     EQ_TAC THEN
     RWTAC [] THENL
     [RES_TAC THEN RWTAC [] THEN METIS_TAC [],
      METIS_TAC [],
      METIS_TAC [],
      METIS_TAC []]]);

val write_reg_wf_thm = Q.prove (
`!ii r x. wfM ii (write_reg ii r x)`,
RWTAC [write_reg_def, write_reg_ev_def, write_location_ev_def, wfM_def, 
       wf_def, next_eiid_def] THEN
FSTAC [reads_def, writes_def] THEN
RWTAC [acyclic_def, sTC_EMPTY, DOM_def, SUBSET_DEF, RANGE_def] THEN
FSTAC [proc_def, loc_def] THEN
RWTAC [EXTENSION]);

val read_reg_wf_thm = Q.prove (
`!ii r. wfM ii (read_reg ii r)`,
RWTAC [read_reg_def, read_reg_ev_def, read_location_ev_def, wfM_def, 
       wf_def, next_eiid_def] THEN
FSTAC [writes_def] THEN
RWTAC [acyclic_def, sTC_EMPTY, DOM_def, SUBSET_DEF, RANGE_def] THEN
FSTAC [proc_def, loc_def] THEN
RWTAC [EXTENSION]);

val write_eip_wf_thm = Q.prove (
`!ii x. wfM ii (write_eip ii x)`,
RWTAC [write_eip_def, write_eip_ev_def, write_location_ev_def, wfM_def, 
       wf_def, next_eiid_def] THEN
FSTAC [reads_def, writes_def] THEN
RWTAC [acyclic_def, sTC_EMPTY, DOM_def, SUBSET_DEF, RANGE_def] THEN
FSTAC [proc_def, loc_def] THEN
RWTAC [EXTENSION]);

val read_eip_wf_thm = Q.prove (
`!ii. wfM ii (read_eip ii)`,
RWTAC [read_eip_def, read_eip_ev_def, read_location_ev_def, wfM_def, 
       wf_def, next_eiid_def] THEN
FSTAC [writes_def] THEN
RWTAC [acyclic_def, sTC_EMPTY, DOM_def, SUBSET_DEF, RANGE_def] THEN
FSTAC [proc_def, loc_def] THEN
RWTAC [EXTENSION]);

val write_eflag_wf_thm = Q.prove (
`!ii f bo. wfM ii (write_eflag ii f bo)`,
Cases_on `bo` THEN 
RWTAC [write_eflag_def, write_eflag_ev_def, write_location_ev_def] THENL
[MATCH_MP_TAC choiceT_wf_thm,
 ALL_TAC,
 ALL_TAC] THEN
RWTAC [wfM_def, wf_def, reads_def, writes_def, DOM_def, RANGE_def, SUBSET_DEF] THEN
RWTAC [acyclic_def, sTC_EMPTY] THEN
FSTAC [proc_def, loc_def, next_eiid_def] THEN
RWTAC [EXTENSION] THEN
FSTAC []);

val read_eflag_wf_thm = Q.prove (
`!ii f. wfM ii (read_eflag ii f)`,
RWTAC [read_eflag_def, read_eflag_ev_def, read_location_ev_def] THEN
MATCH_MP_TAC mapT_wf_thm THEN
RWTAC [wfM_def, wf_def, reads_def, writes_def, DOM_def, RANGE_def, SUBSET_DEF] THEN
RWTAC [acyclic_def, sTC_EMPTY] THEN
FSTAC [proc_def, loc_def, next_eiid_def] THEN
RWTAC [EXTENSION] THEN
FSTAC []);

val write_m32_wf_thm = Q.prove (
`!ii a x. wfM ii (write_m32 ii a x)`,
RWTAC [write_m32_def, write_m32_ev_def, failureT_ev_def, write_location_ev_def,
       wfM_def, wf_def, next_eiid_def] THEN
FSTAC [writes_def] THEN
RWTAC [acyclic_def, sTC_EMPTY, DOM_def, SUBSET_DEF, RANGE_def] THEN
FSTAC [proc_def, loc_def] THEN
RWTAC [EXTENSION]);

val read_m32_wf_thm = Q.prove (
`!ii a. wfM ii (read_m32 ii a)`,
RWTAC [read_m32_def, read_m32_ev_def, failureT_ev_def, read_location_ev_def,
       wfM_def, wf_def, next_eiid_def] THEN
FSTAC [writes_def] THEN
RWTAC [acyclic_def, sTC_EMPTY, DOM_def, SUBSET_DEF, RANGE_def] THEN
FSTAC [proc_def, loc_def] THEN
RWTAC [EXTENSION]);

val T = 
  let 
    val step =
      (MAP_FIRST MATCH_MP_TAC 
           [if_wf_thm, seqT_wf_ctxt_thm, parT_wf_thm, parT_unit_wf_thm, addT_wf_thm, 
            choiceT_wf_thm, mapT_wf_thm] ORELSE
       MAP_FIRST MATCH_ACCEPT_TAC 
          [constT_wf_thm, read_reg_wf_thm, read_m32_wf_thm, write_m32_wf_thm, 
           write_reg_wf_thm, failureT_wf_thm, write_eflag_wf_thm,
           write_eip_wf_thm, read_eip_wf_thm]) THEN
      (REPEAT STRIP_TAC) THEN
      BETA_TAC;
    fun T x = 
      ((step THEN T) ORELSE ALL_TAC) x
  in
    T
  end;

val ea_Xr_wf_thm = Q.prove (
`!iiid r. wfM iiid (ea_Xr r)`,
RWTAC [ea_Xr_def] THEN
T);

val ea_Xi_wf_thm = Q.prove (
`!iiid i. wfM iiid (ea_Xi i)`,
RWTAC [ea_Xi_def] THEN
T);

val ea_Xrm_base_wf_thm = Q.prove (
`!ii ro. wfM ii (ea_Xrm_base ii ro)`,
Cases_on `ro` THEN
RWTAC [ea_Xrm_base_def] THEN
T);

val ea_Xrm_index_wf_thm = Q.prove (
`!ii ro. wfM ii (ea_Xrm_index ii ro)`,
Cases_on `ro` THEN
RWTAC [ea_Xrm_index_def] THEN
T THEN
Cases_on `x` THEN
RWTAC [ea_Xrm_index_def] THEN
T THEN
DO_INM THEN
FSTAC [mem_store_def]);

val ea_Xrm_wf_thm = Q.prove (
`!ii X. wfM ii (ea_Xrm ii X)`,
Cases_on `X` THEN
RWTAC [ea_Xrm_def] THEN
T THENL
[RWTAC [ea_Xr_wf_thm],
 RWTAC [ea_Xrm_index_wf_thm],
 RWTAC [ea_Xrm_base_wf_thm],
 METIS_TAC [ea_Xrm_index_inM_thm, ea_Xrm_base_inM_thm, read_reg_inM_thm],
 METIS_TAC [ea_Xrm_index_inM_thm, ea_Xrm_base_inM_thm, read_reg_inM_thm],
 Cases_on `x` THEN
     RWTAC [constT_wf_thm],
 DO_INM THEN
     FSTAC [mem_store_def]]);

val ea_Xdest_wf_thm = Q.prove (
`!ii X0. wfM ii (ea_Xdest ii X0)`,
Cases_on `X0` THEN
RWTAC [ea_Xdest_def, ea_Xrm_wf_thm, ea_Xr_wf_thm]);

val ea_Xsrc_wf_thm = Q.prove (
`!ii X0. wfM ii (ea_Xsrc ii X0)`,
Cases_on `X0` THEN
RWTAC [ea_Xsrc_def, ea_Xi_wf_thm, ea_Xrm_wf_thm, ea_Xr_wf_thm]);

val ea_Ximm_rm_wf_thm = Q.prove (
`!ii X0. wfM ii (ea_Ximm_rm ii X0)`,
Cases_on `X0` THEN
RWTAC [ea_Ximm_rm_def, ea_Xi_wf_thm, ea_Xrm_wf_thm]);

val read_ea_wf_thm = Q.prove (
`!ii ea. wfM ii (read_ea ii ea)`,
Cases_on `ea` THEN
RWTAC [read_ea_def] THEN
T);

val read_src_ea_wf_thm = Q.prove (
`!ii ds. wfM ii (read_src_ea ii ds)`,
RWTAC [read_src_ea_def] THEN
T THEN
RWTAC [ea_Xsrc_wf_thm, read_ea_wf_thm] THEN
DO_INM THEN
FSTAC [mem_store_def]);

val read_dest_ea_wf_thm = Q.prove (
`!ii ds. wfM ii (read_dest_ea ii ds)`,
RWTAC [read_dest_ea_def] THEN
T THEN
RWTAC [ea_Xdest_wf_thm, read_ea_wf_thm] THEN
DO_INM THEN
FSTAC [mem_store_def]);

val write_ea_wf_thm = Q.prove (
`!ii ea x. wfM ii (write_ea ii ea x)`,
Cases_on `ea` THEN
RWTAC [write_ea_def] THEN
T);

val jump_to_ea_wf_thm = Q.prove (
`!ii eip X. wfM ii (jump_to_ea ii eip X)`,
Cases_on `X` THEN
RWTAC [jump_to_ea_def] THEN
T THEN
DO_INM THEN
FSTAC [mem_store_def]);

val write_arith_eflags_except_CF_wf_thm = Q.prove (
`!ii w. wfM ii (write_arith_eflags_except_CF ii w)`,
RWTAC [write_arith_eflags_except_CF_def, write_PF_def, write_ZF_def, write_SF_def] THEN
T THEN
DO_INM THEN
RWTAC [] THEN
FSTAC [] THEN
RWTAC []);

val write_logical_eflags_wf_thm = Q.prove (
`!ii w c. wfM ii (write_logical_eflags ii w)`,
RWTAC [write_logical_eflags_def, write_PF_def, write_ZF_def, write_SF_def] THEN
T THEN
DO_INM THEN
RWTAC [] THEN
FSTAC [] THEN 
RWTAC []);

val write_arith_eflags_wf_thm = Q.prove (
`!ii w c. wfM ii (write_arith_eflags ii (w, c))`,
RWTAC [write_arith_eflags_def] THEN
T THEN
DO_INM THEN
RWTAC [write_arith_eflags_except_CF_wf_thm] THEN
FSTAC [] THEN
RWTAC []);

val erase_eflags_wf_thm = Q.prove (
`!ii. wfM ii (erase_eflags ii)`,
RWTAC [erase_eflags_def] THEN
T THEN
DO_INM THEN
FSTAC [] THEN
RWTAC []);

val write_binop_wf_thm = Q.prove (
`!ii X x y ea. wfM ii (write_binop ii X x y ea)`,
Cases_on `X` THEN
RWTAC [write_binop_def, write_arith_result_def, write_arith_result_no_write_def,
       write_logical_result_def, write_result_erase_eflags_def, write_logical_result_no_write_def, 
       add_with_carry_out_def, sub_with_borrow_out_def] THEN
T THEN
RWTAC [write_arith_eflags_wf_thm, erase_eflags_wf_thm, write_ea_wf_thm, write_logical_eflags_wf_thm] THEN
DO_INM THEN
RWTAC [] THEN
FSTAC [] THEN
RWTAC []);

val write_monop_wf_thm = Q.prove (
`!ii X x ea. wfM ii (write_monop ii X x ea)`,
Cases_on `X` THEN
RWTAC [write_monop_def, write_arith_result_no_CF_def] THEN
T THEN
RWTAC [write_arith_eflags_except_CF_wf_thm, write_ea_wf_thm] THEN
DO_INM THEN
RWTAC [] THEN
FSTAC [] THEN
RWTAC []);

val read_cond_wf_thm = Q.prove (
`!ii x. wfM ii (read_cond ii x)`,
Cases_on `x` THEN
RWTAC [read_cond_def] THEN
T THEN
RWTAC [read_eflag_wf_thm] THEN
DO_INM THEN
FSTAC [mem_store_def]);

val x86_exec_pop_wf_thm = Q.prove (
`!ii rm. wfM ii (x86_exec_pop ii rm)`,
RWTAC [x86_exec_pop_def] THEN
T THEN
DO_INM THEN
FSTAC [mem_store_def] THEN
Cases_on `x` THEN
FSTAC [] THEN
T THEN
RWTAC [ea_Xrm_wf_thm] THEN
DO_INM THEN
FSTAC [mem_store_def] THEN
RWTAC [] THEN
Cases_on `x` THEN
RWTAC [] THEN
METIS_TAC [write_ea_wf_thm]);

val x86_exec_push_wf_thm = Q.prove (
`!ii imm_rm. wfM ii (x86_exec_push ii imm_rm)`,
RWTAC [x86_exec_push_def] THEN
T THEN
RWTAC [ea_Ximm_rm_wf_thm, read_ea_wf_thm] THEN
DO_INM THEN
FSTAC [mem_store_def] THEN
RWTAC [] THEN
Cases_on `x` THEN
RWTAC [] THEN
T THEN
DO_INM THEN
FSTAC []);

val x86_exec_pop_eip_wf_thm = Q.prove (
`!ii. wfM ii (x86_exec_pop_eip ii)`,
RWTAC [x86_exec_pop_eip_def] THEN
T THEN
DO_INM THEN
FSTAC [mem_store_def] THEN
Cases_on `x` THEN
FSTAC [] THEN
T THEN
DO_INM THEN
FSTAC [mem_store_def]);

val x86_exec_push_eip_wf_thm = Q.prove (
`!ii. wfM ii (x86_exec_push_eip ii)`,
RWTAC [x86_exec_push_eip_def] THEN
T THEN
DO_INM THEN
FSTAC [mem_store_def] THEN
RWTAC [] THEN
Cases_on `x` THEN
RWTAC [] THEN
T THEN
DO_INM THEN
FSTAC []);

val call_dest_from_ea_wf_thm = Q.prove (
`!ii eip Xea. wfM ii (call_dest_from_ea ii eip Xea)`,
Cases_on `Xea` THEN
RWTAC [call_dest_from_ea_def] THEN
T);

val wfthms = [read_src_ea_wf_thm, read_dest_ea_wf_thm, ea_Xrm_wf_thm,
read_ea_wf_thm, write_ea_wf_thm, write_binop_wf_thm, write_monop_wf_thm,
ea_Ximm_rm_wf_thm, ea_Xdest_wf_thm, x86_exec_pop_wf_thm, x86_exec_push_wf_thm, 
read_cond_wf_thm, jump_to_ea_wf_thm, x86_exec_pop_eip_wf_thm,
x86_exec_push_eip_wf_thm, call_dest_from_ea_wf_thm, ea_Xsrc_wf_thm];

val well_formedM_def = Define `
  well_formedM id m =
    !eiid_start eiid_next x E. 
      (eiid_next, x, E) IN m eiid_start 
      ==> 
      well_formed_event_structure E /\
      (E.procs = { e.iiid.proc | e IN E.events }) /\
      (!e. e IN E.events ==> (e.iiid = id)) /\
      (!e. e IN E.events ==> ~(e.eiid IN eiid_start)) /\
      (eiid_next = eiid_start UNION {e.eiid | e IN E.events})`;

val well_formedM_thm = Q.prove (
`!ii m. wfM ii m ==> well_formedM ii m`,
RWTAC [wfM_def, well_formedM_def] THEN
METIS_TAC [wf_thm, RQ wf10]);

val es_lock_well_formed_thm = Q.prove (
`!E.
  (!e1 e2. e1 IN E.events /\ e2 IN E.events ==> (e1.iiid = e2.iiid)) /\
  (?e. e IN E.events /\ mem_load e) /\
  well_formed_event_structure E 
  ==>
  well_formed_event_structure (event_structure_lock E)`,
RWTAC [well_formed_event_structure_def, event_structure_lock_def] THENL
[METIS_TAC [],
 FSTAC [writes_def, reads_def] THEN
     METIS_TAC [],
 FSTAC [writes_def, reads_def] THEN
     METIS_TAC [],
 RWTAC [PER_def]]);

val lockT_well_formed_thm = Q.prove (
`!ii m. 
  wfM ii m /\ 
  existsM mem_load m
  ==> 
  well_formedM ii (lockT m)`,
RWTAC [wfM_def, well_formedM_def, lockT_def, lockT_ev_def, LET_THM, existsM_def] THEN
Cases_on `x'` THEN 
Cases_on `r` THEN 
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN1
METIS_TAC [es_lock_well_formed_thm, wf_thm] THEN
FSTAC [event_structure_lock_def] THEN
METIS_TAC [RQ wf10]);

val lockT_well_formed_thm2 = Q.prove (
`!ii m. 
  well_formedM ii m /\
  existsM mem_load m
  ==>
  well_formedM ii (lockT m)`,
RWTAC [well_formedM_def, lockT_def, lockT_ev_def, LET_THM, existsM_def] THEN
Cases_on `x'` THEN 
Cases_on `r` THEN 
FSTAC [] THEN
RWTAC [] THEN
FSTAC [] THEN1
METIS_TAC [es_lock_well_formed_thm] THEN
FSTAC [event_structure_lock_def] THEN
METIS_TAC []);

val TAC =
RWTAC [x86_exec_def, bump_eip_def, LAMBDA_PROD2, x86_exec_call_def, x86_exec_ret_def,
       x86_exec_pushad_def, x86_exec_popad_def, LET_THM] THEN
T THEN
FSTAC [LAMBDA_PROD2] THEN
TRY (MAP_FIRST MATCH_ACCEPT_TAC wfthms) THEN
DO_INM THEN
RWTAC [] THEN
FSTAC [mem_store_def] THEN
RWTAC [];

val ea_Xrm_lem1 = Q.prove (
`!eiid_next x E ii r eiid_start.
  (eiid_next, x, E) IN ea_Xrm ii (Xr r) eiid_start
  ==>
  (x = Xea_r r)`,
RWTAC [ea_Xrm_def, ea_Xr_def, constT_def, constT_ev_def]);

val ea_Xrm_lem2 = Q.prove (
`!eiid_next x E ii i b d eiid_start.
  (eiid_next, x, E) IN ea_Xrm ii (Xm i b d) eiid_start
  ==>
  ?v. x = Xea_m v`,
RWTAC [ea_Xrm_def, parT_def, parT_ev_def, seqT_def, seqT_ev_def, constT_def,
       constT_ev_def, LET_THM] THEN
FSTAC [] THEN
RWTAC [] THEN
Cases_on `x'` THEN
FSTAC []);

val ea_Xrm_lem3 = Q.prove (
`!x X eiid_next E eiid_start. 
  rm_is_memory_access X /\
  (eiid_next,x,E) IN ea_Xrm ii X eiid_start
  ==> 
  (?a. x = Xea_m a)`,
Cases_on `X` THEN
RWTAC [rm_is_memory_access_def] THEN
METIS_TAC [ea_Xrm_lem2]);

val TAC2 =
FSTAC [parT_def, parT_ev_def, seqT_def, seqT_ev_def, addT_def, addT_ev_def, 
       constT_def, constT_ev_def, LET_THM, parT_unit_def, parT_unit_ev_def] THEN
NTAC 2 (FSTAC [] THEN RWTAC []);

fun DISJN_TAC 1 = DISJ1_TAC
  | DISJN_TAC 2 = DISJ2_TAC THEN (TRY DISJ1_TAC)
  | DISJN_TAC n = DISJ2_TAC THEN DISJN_TAC (n - 1);

val instr_wf_thm = Q.prove (
`!ii Xinst len. well_formedM ii (x86_exec ii Xinst len)`,
REPEAT STRIP_TAC THEN
Cases_on `Xinst` THENL
[MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC THEN
     TAC2,
 RWTAC [x86_exec_def] THEN
     TRY (MATCH_MP_TAC lockT_well_formed_thm) THEN
     TRY (MATCH_MP_TAC well_formedM_thm) THEN
     RWTAC [bump_eip_def] THENL
     [FSTAC [rm_is_memory_access_def],
      FSTAC [rm_is_memory_access_def],
      TAC THEN
          TAC2 THEN
          Cases_on `X'` THEN
          FSTAC [rm_is_memory_access_def] THEN
          Cases_on `x''''` THEN
          Cases_on `r` THEN
          FSTAC [] THEN
          RWTAC [] THEN
          FSTAC [] THEN
          RWTAC [] THEN
          IMP_RES_TAC ea_Xrm_lem2 THEN
          FSTAC [],
      DO_EXISTSM THEN
          RWTAC [] THEN
          DO_EXISTSM THEN
          RWTAC [mem_load_def] THEN
          METIS_TAC [ea_Xrm_lem3],
      TAC,
      TAC THEN
          TAC2 THEN
          Cases_on `X'` THEN
          FSTAC [rm_is_memory_access_def] THEN
          Cases_on `x''''` THEN
          Cases_on `r` THEN
          FSTAC [] THEN
          RWTAC [] THEN
          FSTAC [] THEN
          RWTAC [] THEN
          IMP_RES_TAC ea_Xrm_lem1 THEN
          FSTAC []],
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC,
 MATCH_MP_TAC well_formedM_thm THEN
     TAC]);

val locked_has_mem_load_instr_def = Define `
  (locked_has_mem_load_instr (Xprefix Xlock g2 i) = x86_lock_ok i) /\
  (locked_has_mem_load_instr (Xprefix Xg1_none g2 i) = T)`;

val TAC3 =
RWTAC [x86_exec_def, bump_eip_def, LAMBDA_PROD2] THEN
DO_EXISTSM THEN
RWTAC [mem_load_def] THEN
Cases_on `X0` THEN
FSTAC [dest_accesses_memory_def, ea_Xdest_def] THEN
METIS_TAC [ea_Xrm_lem3];

val TAC4 =
RWTAC [x86_exec_def, bump_eip_def, LAMBDA_PROD2] THEN
DO_EXISTSM THEN
RWTAC [] THEN
DO_EXISTSM THEN
RWTAC [mem_load_def] THEN
FSTAC [ea_Xdest_def] THEN
METIS_TAC [ea_Xrm_lem3];

val x86_execute_well_formedM_thm = Q.store_thm ("x86_execute_well_formedM_thm",
`!ii Xinst len. 
  locked_has_mem_load_instr Xinst
  ==>
  well_formedM ii (x86_execute ii Xinst len)`,
Cases_on `Xinst` THEN
Cases_on `X'` THEN
RWTAC [x86_execute_def, locked_has_mem_load_instr_def] THENL
[MATCH_MP_TAC lockT_well_formed_thm2 THEN
     RWTAC [] THEN1
     METIS_TAC [instr_wf_thm] THEN
     Cases_on `X1` THEN
     FSTAC [x86_lock_ok_def] THEN
     RWTAC [] THENL
     [TAC3,
      TAC3,
      TAC3,
      TAC3,
      TAC3,
      TAC4,
      TAC4,
      TAC4,
      TAC4,
      TAC4,
      TAC4,
      RWTAC [x86_exec_def] THEN1
          FSTAC [rm_is_memory_access_def] THEN
          TAC4],
 METIS_TAC [instr_wf_thm]]);

val es_bigunion_well_formed_thm = Q.store_thm ("es_bigunion_well_formed_thm",
`!Es. 
  (!E E' e e'. 
    E IN Es /\ E' IN Es /\ e IN E.events /\ e' IN E'.events /\ (e.iiid = e'.iiid)
    ==>
    (E = E')) /\
  FINITE (BIGUNION {es.procs | es IN Es}) /\
  (!E. E IN Es ==> well_formed_event_structure E)
  ==>
  well_formed_event_structure (event_structure_bigunion Es)`,
RWTAC [event_structure_bigunion_def, well_formed_event_structure_def] THENL
[Cases_on `?E. E IN Es /\ ?e. e IN E.events /\ (e.iiid = iiid)` THEN
     FSTAC [] THENL
     [RWTAC [] THEN
          `{eiid | ?e'. (?s. e' IN s /\ ?es. (s = es.events) /\ es IN Es) /\ (e'.iiid = e.iiid) /\ (e'.eiid = eiid)} =
           {eiid | ?e'. e' IN E.events /\ (e'.iiid = e.iiid) /\ (e'.eiid = eiid)}` by
                    (RWTAC [EXTENSION] THEN
                     METIS_TAC []) THEN
          METIS_TAC [],
      `{eiid | ?e. (?s. e IN s /\ ?es. (s = es.events) /\ es IN Es) /\ (e.iiid = iiid) /\ (e.eiid = eiid)} = {}` by
                    (RWTAC [EXTENSION] THEN
                     METIS_TAC []) THEN
          METIS_TAC [FINITE_EMPTY]],
 METIS_TAC [],
 METIS_TAC [],
 FSTAC [SUBSET_DEF, DOM_def] THEN
     METIS_TAC [],
 FSTAC [SUBSET_DEF, RANGE_def] THEN
     METIS_TAC [],
 MATCH_MP_TAC acyclic_bigunion_thm THEN
     FSTAC [DISJOINT_DEF, SUBSET_DEF, DOM_def, RANGE_def] THEN
     CONJ_TAC THENL
     [RWTAC [EXTENSION] THEN
          METIS_TAC [],
      METIS_TAC []],
 METIS_TAC [],
 METIS_TAC [],
 FSTAC [writes_def, reads_def] THEN
     RWTAC [] THEN
     `es' = es` by METIS_TAC [] THEN
     RWTAC [] THEN
     `(e1, e2) IN sTC (es.intra_causality UNION (BIGUNION {es.intra_causality | es IN Es})) \/ 
      (e2, e1) IN sTC (es.intra_causality UNION (BIGUNION {es.intra_causality | es IN Es}))` 
                 by METIS_TAC [sTC_UNION_lem] THEN
     `es.intra_causality UNION (BIGUNION {es.intra_causality | es IN Es}) =
      BIGUNION {es.intra_causality | es IN Es}`
                 by (RWTAC [EXTENSION] THEN METIS_TAC []) THEN
     FSTAC [],
 FSTAC [writes_def, reads_def] THEN
     RWTAC [] THEN
     `es' = es` by METIS_TAC [] THEN
     RWTAC [] THEN
     `(e1, e2) IN sTC (es.intra_causality UNION (BIGUNION {es.intra_causality | es IN Es})) \/ 
      (e2, e1) IN sTC (es.intra_causality UNION (BIGUNION {es.intra_causality | es IN Es}))` 
                 by METIS_TAC [sTC_UNION_lem] THEN
     `es.intra_causality UNION (BIGUNION {es.intra_causality | es IN Es}) =
      BIGUNION {es.intra_causality | es IN Es}`
                 by (RWTAC [EXTENSION] THEN METIS_TAC []) THEN
     FSTAC [],
 METIS_TAC [],
 FSTAC [PER_def, SUBSET_DEF] THEN
     RWTAC [] THENL
     [METIS_TAC [],
      METIS_TAC [],
      Cases_on `es = es'` THEN1
          METIS_TAC [] THEN
          CCONTR_TAC THEN
          FSTAC [DISJOINT_DEF, EXTENSION] THEN
          RES_TAC],
 METIS_TAC [],
 FSTAC [PER_def, SUBSET_DEF] THEN
     METIS_TAC [],
 METIS_TAC []]);

val _ = export_theory ();
