{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

import System
import System.IO
import Foreign.C
import Thread
import SockIO
import Workers
import System.Posix hiding (Stop)
import Data.List
import Data.IORef
import Data.Maybe
import System.Random
import Numeric

buffer_size = 1600

data LoadGen = LoadGen
       { lg_n_cpu       :: !Int
       , lg_pool_size   :: !Int
       , lg_n_conn      :: !Int
       , lg_range       :: !Int
       , lg_data_mb     :: !Integer
       , lg_hostname    :: !String
       , lg_port        :: !Int
       , lg_prefix      :: !String
       , lg_postfix     :: !String
       , lg_todo        :: TVar Integer
       } deriving Show

instance Show (TVar a) where 
  show t = "[TVar]"

-------------- Helper functions ------------------------
parse_int arg = let [(i,rem)] = readDec arg in i

prints fd s = sys_nbio $ putStrLn $ "[" ++ (show fd) ++ "] " ++ s

roll_dice :: Int -> CPSMonad Int
roll_dice mx = sys_nbio $ getStdRandom (randomR (0,mx-1))
 

-------------- Each thread -----------------------------

generator state = do 
   loop $
      sys_catch ( do 
          sock <- sock_connect (lg_hostname state) (lg_port state)     

          sys_catch( do
              -- find a random number
              file_no <- roll_dice (lg_range state)
              let filename = create_file_name file_no
              -- issue a minimal HTTP request
              sock_write_line sock $  "GET " ++ filename ++ " HTTP/1.0\r\n"
              -- read the socket until EOF
              loop_until_zero $ do
                 (Chunk _ _ numread) <- sock_recv_any sock
                 dec_and_test numread 
                 return numread
           )(\e-> prints sock e)

          sock_close sock
          return ()
      )(\e-> prints "" e )

 where create_file_name file_no = 
           let s1 = "000000" ++ (show file_no)
               s2 = reverse $ take 6 $ reverse s1
           in "/" ++ (lg_prefix state) ++ s2 ++ (lg_postfix state)

       todo = (lg_todo state)
       
       dec_and_test numread = do 
          b <- sys_nbio $ atomically $ do
              v <- readTVar todo
              writeTVar todo (v-(fromIntegral numread))
              return (v<=0)
          if b then do
              sys_nbio $ putStrLn "terminate!"
              sys_halt	
           else return ()

-----------------------------------------------------------------

main = do 
  args <- getArgs
  if (length args /= 9) then do
       putStrLn $ "Usage: executable +RTS -H20M -N{n_cpu} -RTS n_cpu pool_size n_conn range data hostname port prefix postfix"
       putStrLn $ "  n_cpu:      Number of processors"
       putStrLn $ "  pool_size:  Thread pool size for blocking IO operations"
       putStrLn $ "  n_conn:     Number of concurrent HTTP connections"
       putStrLn $ "  range:      Number of files to be accessed."
       putStrLn $ "  data:       Total amount of data (unit:MB) to download from the server"
       putStrLn $ "For example, the arguments"
       putStrLn $ "  +RTS -H20M -N1 -RTS 1 2 64 60000 100 10.0.8.115 8888 'data/' '.bin'"
       putStrLn $ "will launch 64 concurrent application-level threads and download 100MB data."
       putStrLn $ "Each thread randomly issues one of the following HTTP GET requests:"
       putStrLn $ "  http://10.0.8.115:8888/data/000001.bin"
       putStrLn $ "   ................................."
       putStrLn $ "  http://10.0.8.115:8888/data/059999.bin"
       exitFailure
    else return ()
  let lg1 = LoadGen
       { lg_n_cpu       =  parse_int $ args !! 0
       , lg_pool_size   =  parse_int $ args !! 1
       , lg_n_conn      =  parse_int $ args !! 2
       , lg_range       =  parse_int $ args !! 3
       , lg_data_mb     =  parse_int $ args !! 4
       , lg_hostname    =              args !! 5
       , lg_port        =  parse_int $ args !! 6
       , lg_prefix      =              args !! 7
       , lg_postfix     =              args !! 8
       , lg_todo        = undefined
       }
  tv <- atomically $ newTVar ((lg_data_mb lg1) *1048576)

  let lg = lg1 { lg_todo = tv }
  putStrLn $ (show lg)
  putStrLn $ "Total data=" ++ (show $ lg_data_mb lg) ++ "MB"

  let threads = map (\_ -> generator lg) [1..(lg_n_conn lg)]
  default_scheduler (lg_n_cpu lg) (lg_pool_size lg) threads
