(* An ML function value *)

let succ n = n + 1

(* Here's a simple program that just returns 0 over and over... *)

let repeat = Dsl.run(do U(let rec xs : S(D(int)) = cons([0], xs) in xs))
               
(* Now, let's use an embedded function *)

let repeat = Dsl.run(do U(let succ : D(int) -> D(int) = fun x -> [succ] x in
                          let rec ns : S(D(int)) =
                            cons([0], cons(succ (head ns), tail ns))
                          in
                          ns))

(* Now, let's implement integers with a slightly higher-type fixed point.
   
   These particular examples are actually compiled using an unfold, which 
   results in more efficient (ie, non-memory-leaking) code. 
*)

let ints = Dsl.run(do U(let succ : D(int) -> D(int) = fun x -> [succ] x in
                        let rec ints (n : D(int)) :  S(D(int)) =
                          cons(n, ints(succ n))
                        in
                        ints [0]))



(* Now, let's see how we can implement the Fibonacci numbers as a stream *)

let plus x y = x + y

let fibs = Dsl.run(do U(let plus : D(int) -> D(int) -> D(int) = fun x y -> [plus] x y in 
                        let rec fibs (x : D(int), y : D(int)) :  S(D(int)) =
                          cons(x, fibs (y, (plus x y)))
                        in
                        fibs ([0], [1])))



(* Now let's build a few simple GUIs.

   There's a monad of GUI commands, of type Gui(A). The return for the monad
   is written return(blah), and the monadic binding is written let gui(x) = e in e' 

   To run them, call Dsl.guirun <foo>_example
*)

(* This just makes gui with a label *)
let inject hom = Dsl.curry Dsl.delay_one (Dsl.compose Dsl.snd hom)

let label = inject (Dsl.label `CENTER)

let constant = 
 do U(let constant : D(string) -> S(D(string)) =
        fun msg -> let rec msgs : S(D(string)) = cons(msg, msgs) in msgs
      in
      constant)

let constant_label =
 do U(let label : S(D(string)) -> Gui(one) = {label} in
      let constant : D(string) -> S(D(string)) = {constant} in 
      let constant_label : D(string) -> Gui(one) = 
        fun msg -> (label (constant msg))
      in
      constant_label)

let test_constant_label =
 do U(let constant_label : D(string) -> Gui(one) = {constant_label} in 
      constant_label ["Hello world"])


(* Now, let's combine the label with a timer. We construct a 
   a stream of integers, which counts up every tick of the clock. 

   So, we lift the ML string_of_int function to streams, and then
   create a timer, and pass its output converted to strings to the 
   label function. 
*)

let label_timer_gui =
 do U(let label : S(D(string)) -> Gui(one) = {label} in
      let to_string : D(int) -> D(string) = fun n -> [string_of_int] n in
      let succ : D(int) -> D(int) = fun x -> [succ] x in 
      let rec ints (n : D(int)) : S(D(int)) =
        cons(to_string n, ints (succ n)) in
      label (ints [0]))


(* This creates a button and a label, which changes depending on whether
   the mouse is down or up.
*)
let inject hom = Dsl.curry Dsl.delay_one (Dsl.compose Dsl.snd hom)
let label = inject (Dsl.label `CENTER)
let constant = 
 do U(let constant : D(string) -> S(D(string)) =
        fun msg -> let rec msgs : S(D(string)) = cons(msg, msgs) in msgs
      in
      constant)

let bool_to_string b = if b then "True!" else "False!"
 
let button_gui =
 do U(let label : S(D(string)) -> Gui(one) = {label} in
      let constant : D(string) -> S(D(string)) = {constant} in 
      let button : S(D(string)) -> Gui(S(D(bool))) = {inject Dsl.button} in
      let bool_to_string : D(bool) -> D(string) = fun b -> [bool_to_string] b in 
      let gui(downs : S(D(bool))) = button (constant ["Press me!"]) in
      label (map(bool_to_string, downs)))




(* Now, since our buttons tell us whether the button is down or up,
   they do not directly report click events to us. So if we want
   clicks as events, rather than down/up status, we need to write a
   function to convert this status into a click stream. We can do this
   by keeping track of the previous state, and reporting a click
   whenever the button was down on the previous tick, and is up on the
   current tick.

   This is possibly the simplest example of a pervasive pattern in GUI
   programming, in which a state machine is used to synthesize
   high-level events from lower-level events.
*)
let inject hom = Dsl.curry Dsl.delay_one (Dsl.compose Dsl.snd hom)
let label = inject (Dsl.label `CENTER)
let constant = 
 do U(let constant : D(string) -> S(D(string)) =
        fun msg -> let rec msgs : S(D(string)) = cons(msg, msgs) in msgs
      in
      constant)
let bool_to_string b = if b then "True!" else "False!"

let isclick : bool -> bool -> bool =
  fun current prev -> not current && prev

let clickmachine =
 do U(let rec toclick (prior : D(bool), down : S(D(bool))) : S(D(bool)) =
        let current = head down in 
        cons([isclick] prior current,
             toclick (current, tail down))
      in
      toclick)

let clickmachine2 = 
 do U(let rec toclick (prior : D(bool), down : S(D(bool))) : S(D(bool)) =
        cons([isclick] prior (head down),
             toclick ((head down), tail down))
      in
      toclick)


let clickbutton =
 do U(let button : S(D(string)) -> Gui(S(D(bool))) = {inject Dsl.button} in
      let rec toclick (prior : D(bool), down : S(D(bool))) : S(D(bool)) =
        let current = head down in 
        cons([isclick] prior current,
             toclick (current, tail down))
      in
      let clickbutton : S(D(string)) -> Gui(S(D(bool))) =
        fun msgs ->
          let gui(downs) = button(msgs) in
          return (toclick([false], downs))
      in
      clickbutton)

(* Now we'll test the click button. Note that we have built a 
   first-class new widget. *)

let increment = fun b n -> n + (if b then 1 else 0)


let clickbutton_test0 =
 do U(let clickbutton : S(D(string)) -> Gui(S(D(bool))) = {clickbutton} in
      let constant : D(string) -> S(D(string)) = {constant} in
      let gui bs = clickbutton (constant ["Press me"]) in
      ((return ()) : Gui(one)))

let clickbutton_test = 
 do U(let label : S(D(string)) -> Gui(one) = {label} in
      let clickbutton : S(D(string)) -> Gui(S(D(bool))) = {clickbutton} in
      let constant : D(string) -> S(D(string)) = {constant} in 
      let string_of_int : D(int) -> D(string) = fun n -> [string_of_int] n in
      let increment : D(bool) -> D(int) -> D(int) = fun b n -> [increment] b n in
      let count : S(D(bool)) -> S(D(int)) =
        fun bs ->
          let rec counter (state : S(D(bool)) * D(int)) : S(D(int)) =
            let bs = fst state in
            let n = snd state in
            let bs' : Next(S(D(bool))) = next(tail bs) in 
            let n' = increment (head bs) n in
            cons(n, counter(await(bs'), n'))
          in
          counter(bs, [0])
      in
      let gui(clicks) = clickbutton (constant ["Press me!"]) in
      label (map(string_of_int, count clicks)))

(* Now, a simple calculator application *)

(* First, we'll give the state machine for the calculator. This is a simple 4-function
   calculator, so our state will be a pair, consisting of the current number being
   input-ed, and a register containing the operation and number to apply it to (if
   it is defined). 
*)

type op = int -> int -> int 
type state =
  | Lentry of int * op * int
  | Rentry of int * op * int
  | Noentry of int * op * int
  | Continue of int * op

let init_state = Rentry(0, (+), 0)

(* Next, we'll define the event type, which correspond to the *)

type event = Digit of int | Op of op | Equals | Clear | Nothing

let merge e1 e2 =
  match e1, e2 with
  | Nothing, _ -> e2
  | _ -> e1

(* The transition relation for the calculator. The behavior of a
   simple 4-function calculator is surprisingly irregular, and
   illustrates why it's a good idea to put it all in one place, where
   it can be looked at without the interference of GUI code or
   callbacks to complicate matters.

   In functional programming, it's common to code up state machines as
   sets of mutually-recursive functions. A good question to look at is
   the relationship between these explicit state machines and the
   implicit state machines.

   One surprising aspect of our work is the absence of continuations,
   which prior work on functional GUIs and event loops stressed a lot.
   I suspect that if find a way of writing these state machines as
   recursive functions, we'll need some kind of CPS to be able to to
   break the iteration at each time step and get the right coroutine
   behavior. (This idea is also reminiscent of loops in Esterel.)
*)

let step event state =
  match event, state with
  | Digit n, Rentry(a, op, b)  -> Rentry(a, op, 10*b + n)
  | Equals,  Rentry(a, op, b)  -> Noentry(op a b, op, b)
  | Op op',  Rentry(a, op, b)  -> Continue(op a b, op')
  | Digit n, Noentry(a, op, b) -> Lentry(n, op, b)
  | Equals,  Noentry(a, op, b) -> Noentry(op a b, op, b)
  | Op op',  Noentry(a, op, b) -> Continue(a, op')
  | Digit n, Lentry(a, op, b)  -> Lentry(10*a + n, op, b)
  | Equals,  Lentry(a, op, b)  -> Noentry(op a b, op, b)
  | Op op',  Lentry(a, op, b)  -> Continue(op a b, op')
  | Digit n, Continue(a, op)   -> Rentry(a, op, n)
  | Equals,  Continue(a, op)   -> Noentry(op a a, op, a)
  | Op op',  Continue(a, op)   -> Continue(a, op')
  | Clear,   state             -> init_state
  | Nothing, state             -> state

let display : state -> string = function
   | Continue(n, _)
   | Noentry(n, _, _)
   | Rentry(_, _, n) 
   | Lentry(n, _, _) -> string_of_int n

let div a b = if b = 0 then 0 else a / b

let calculator_button =
 do U(let constant : D(int) -> S(D(string)) = {constant} in
      let clickbutton : S(D(string)) -> Gui(S(D(bool))) = {clickbutton} in 
      let event_when : D(event) -> D(bool) -> D(event) =
        fun event b -> [fun e b -> if b then e else Nothing] event b in
      let cbutton : D(string) -> D(event) -> Gui(S(D(event))) =
        fun msg event -> 
          let gui(clicks) = clickbutton (constant msg) in
          return(map(event_when event, clicks))
      in
      cbutton)

let numeric_button =
 do U(let constant : D(int) -> S(D(string)) = {constant} in
      let button : D(string) -> D(event) -> Gui(S(D(event))) = {calculator_button} in 
      let string_of_int : D(int) -> D(string) = fun n -> [string_of_int] n in
      let event_of_int : D(int) -> D(event) = fun n -> [fun n -> Digit n] n in
      let nbutton : D(int) -> Gui(S(D(event))) =
        fun n -> button (string_of_int n) (event_of_int n)
      in
      nbutton)

let hstack () = inject (Dsl.stack `HORIZONTAL `HOMOGENEOUS `START `EXPAND)
let vstack () = inject (Dsl.stack `VERTICAL `HOMOGENEOUS `START `EXPAND)
let vstack2 ()  = inject (Dsl.stack `VERTICAL `INHOMOGENEOUS `FINISH `NOEXPAND)

let button_row =
 do U(let hstack : Gui(S(D(event))) -> Gui(S(D(event))) = {hstack ()} in 
      let merge : (D(event) * D(event)) -> D(event) =
        fun ee' -> [merge] (fst ee') (snd ee')
      in
      let row : Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) =
        fun a b c d ->
          hstack(let gui(es1) = a in
                 let gui(es2) = b in
                 let gui(es3) = c in
                 let gui(es4) = d in
                 return(map(merge, zip(es1,
                        map(merge, zip(es2,
                        map(merge, zip(es3,
                                       es4))))))))
      in
      row)   

let row_stack = 
 do U(let vstack : Gui(S(D(event))) -> Gui(S(D(event))) = {vstack ()} in 
      let merge : (D(event) * D(event)) -> D(event) =
        fun ee' -> [merge] (fst ee') (snd ee')
      in
      let row : Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) =
        fun a b c d ->
          vstack(let gui(es1) = a in
                 let gui(es2) = b in
                 let gui(es3) = c in
                 let gui(es4) = d in
                 return(map(merge, zip(es1,
                        map(merge, zip(es2,
                        map(merge, zip(es3,
                                       es4))))))))
      in
      row)

let button_layout =
 do U(let cbutton : D(string) -> D(event) -> Gui(S(D(event))) = {calculator_button} in
      let nbutton : D(int) -> Gui(S(D(int))) = {numeric_button} in
      let button_row : Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) = {button_row} in
      let row_stack : Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) -> 
                Gui(S(D(event))) = {row_stack} in
      row_stack
        (button_row
           (nbutton [1])
           (nbutton [2])
           (nbutton [3])
           (cbutton ["X"] [Op ( * )]))
        (button_row
           (nbutton [4])
           (nbutton [5])
           (nbutton [6])
           (cbutton ["-"] [Op ( - )]))
        (button_row
           (nbutton [7])
           (nbutton [8])
           (nbutton [9])
           (cbutton ["+"] [Op ( + )]))
        (button_row
           (cbutton ["C"] [Clear])
           (nbutton [0])
           (cbutton ["="] [Equals])
           (cbutton ["/"] [Op div])))

let calculator_gui =
 do U(let label : S(D(string)) -> Gui(one) = {label} in
      let button_layout : Gui(S(D(event))) = {button_layout} in
      let vstack2 : Gui(one) -> Gui(one) = {vstack2 ()} in
      let step : D(event) -> D(state) -> D(state) = fun e s -> [step] e s in
      let display : D(state) -> D(string) = fun s -> [display] s in 
      let rec states (state : D(state), events : S(D(event))) : S(D(state)) =
        let nextstate = step (head events) state in 
        cons(state, states(nextstate, tail(events)))
      in
      vstack2(let gui(events) = button_layout in
              label(map(display, states([init_state], events)))))


                       
(* **************** RPN **************** *)

module RPN = 
struct
  type stack = int list
  type op = Num of int | Push | Pop | Plus | Minus | Times | Clear | Nothing
  
  let eval op stack =
    match (op, stack) with
    | Num n, m :: s        -> (n + 10 * m) :: s
    | Num n, []            -> [n]
    | Push,    s           -> 0 :: s
    | Pop,     _ :: s      -> s
    | Plus,    n :: m :: s -> (n + m) :: s
    | Times,   n :: m :: s -> (n * m) :: s
    | Minus,   n :: m :: s -> (n - m) :: s
    | Clear,  n :: s       -> 0 :: s
    | _,       s           -> s
  
  let rec display = function
    | [] -> "0"
    | [n] -> string_of_int n
    | n :: s -> Printf.sprintf "%s, %d" (display s) n
  
  let mux o1 o2 =
    match o1, o2 with 
    | Nothing, op -> op
    | op, _       -> op
  
  let bool_to_op op b = if b then op else Nothing 
  
  let stacks =
   do U(let rec stacks (state : D(stack), ops : S(D(op))) : S(D(stack)) =
  	let nextstate : D(stack) = [eval] (head ops) state in 
  	cons(state, stacks(nextstate, tail ops))
        in
        stacks)
  
  let rpn_button =
   do U(let constant : D(string) -> S(D(string)) = {constant} in
        let clickbutton : S(D(string)) -> Gui(S(D(bool))) = {clickbutton} in 
        let bool_to_op : D(op) -> D(bool) -> D(op) =
          fun op b -> [bool_to_op] op b in
        let cbutton : D(string) -> D(op) -> Gui(S(D(op))) =
          fun msg op -> 
            let gui(clicks) = clickbutton (constant msg) in
            return(map(bool_to_op op, clicks))
        in
        cbutton)
  
  let numeric =
   do U(let button : D(string) -> D(op) -> Gui(S(D(op))) = {rpn_button} in 
        let numeric : D(int) -> Gui(S(D(op))) =
          fun n -> button ([string_of_int] n) ([fun n -> Num n] n)
        in
        numeric)
  
  let pack =
   do U(let mux : (D(op) * D(op)) -> D(op) =
          fun ee' -> [mux] (fst ee') (snd ee')
        in
        let pack : (Gui(S(D(op))) -> Gui(S(D(op)))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) =
          fun stack a b c d ->
            stack(let gui(es1) = a in
                  let gui(es2) = b in
                  let gui(es3) = c in
                  let gui(es4) = d in
                  return(map(mux, zip(es1,
                         map(mux, zip(es2,
                         map(mux, zip(es3,
                                      es4))))))))
        in
        pack)
       
  let input =
   do U(let cbutton : D(string) -> D(op) -> Gui(S(D(op))) = {rpn_button} in
        let numeric : D(int) -> Gui(S(D(int))) = {numeric} in
        let vstack : Gui(S(D(event))) -> Gui(S(D(event))) = {vstack ()} in 
        let hstack : Gui(S(D(event))) -> Gui(S(D(event))) = {hstack ()} in 
        let pack : (Gui(S(D(op))) -> Gui(S(D(op)))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) -> 
                   Gui(S(D(op))) = {pack} in 
        pack vstack
          (pack hstack
             (numeric [1])
             (numeric [2])
             (numeric [3])
             (cbutton ["+"] [Plus]))
          (pack hstack
             (numeric [4])
             (numeric [5])
             (numeric [6])
             (cbutton ["-"] [Minus]))
          (pack hstack
             (numeric [7])
             (numeric [8])
             (numeric [9])
             (cbutton ["x"] [Times]))
          (pack hstack
             (cbutton ["D"] [Pop])
             (numeric [0])
             (cbutton [","] [Push])
             (cbutton ["C"] [Clear])))
  
let gui =
   do U(let label : S(D(string)) -> Gui(one) = {label} in
        let input : Gui(S(D(event))) = {input} in
        let vstack2 : Gui(one) -> Gui(one) = {vstack2 ()} in
        let stacks : (D(stack) * S(D(op))) -> S(D(stack)) = {stacks} in
	let display : D(stack) -> D(string) = fun s -> [display] s 
        in
        vstack2(let gui(events) = input in
                label(map(display, stacks([[]], events)))))
end    
          

