(* This module extends the datflow expression monad with some reader state 
   to make everything parametric in the clock and the update list. *)
module Code =
struct
  module D = Dataflow.Expr

  type clock = unit Dataflow.cell
  type updates = unit D.t list ref
  type state = clock * updates
  type 'a code = state -> 'a D.t
  type 'a cell = 'a Dataflow.cell

  let return x s = D.return x 

  let read cell s = D.read cell
  let (>>=) m f s = D.bind (m s) (fun v -> f v s)
  let newref v s = D.local (fun () -> ref v)
  let get r s = D.local (fun () -> !r)
  let set r v s = D.local (fun () -> r := v)
  let local thunk s = D.local thunk

  let get_clock (clock, _) = D.return clock


  let cell code s = D.newcell (code s)
  let clock (clock, _) = D.read clock

  let n = ref 0

  let register label cell (_, updates) =
    let poke = D.bind (D.read cell) (fun _ -> D.return ()) in
    incr n;
    Printf.printf "registered %d in %s\n" !n label;
    flush stdout;
    updates := poke :: !updates;
    D.return ()

 (* Some utility functions for dealing with optionals *)
  let ozip = function 
    | Some x, Some y -> Some(x,y)
    | _ -> None

  let omap f = function
    | None -> None
    | Some v -> Some(f v)

  let ofold none some = function
    | None -> none
    | Some v -> some v

  let (>>-) m f = m >>= (function None -> return None
                         | Some v -> f v)

  let (>>!) m f = m >>= (function None -> assert false
                         | Some v -> f v)
end

open Code

type ('a,'b) hom = 'a -> 'b code

type one = unit
type ('a,'b) prod = 'a * 'b
type 'a next = 'a code
type 'a stream = 'a cell
type ('a,'b) sum = Inl of 'a | Inr of 'b
type 'a discrete = 'a

type ('a,'b) iso = {into : ('a,'b) hom; from : ('b,'a) hom}
type 'a delay = {delay : ('a,'a next) hom;
		 await : 'a -> 'a code code} 

type ('a,'b,'c) closure = {
  env: 'c;
  hom: (('c,'a) prod, 'b) hom;
  op: 'c delay
}
and ('a,'b,'c) closure_handler = {handler: 'd. ('a,'b,'d) closure -> 'c}
and ('a,'b) exp = {unpack: 'result. ('a,'b,'result) closure_handler -> 'result}
let build_exp  closure = {unpack = fun cont -> cont.handler closure}

let delay f = f.delay
let await f = f.await

(* Basic operations upon homs *)

let id x = return x
let compose f g = fun a -> (f a) >>= g

(* Isomorphism operations *)

let into iso = iso.into
let from iso = iso.from

let id_iso = {into = id; from = id}
let compose_iso f g = {into = compose (into f) (into g);
		       from = compose (from g) (from f)}

(* The next type constructor *)

let delay_discrete = 
  let f x = return (return x) in
  {delay = f; await = f}

let delay_one =
  let f = fun () -> return (return ()) in
  {delay = f; await = f}

let delay_prod a b =
  let f aop bop (a,b) = 
    (aop a) >>= (fun a' -> 
    (bop b) >>= (fun b' ->
    return (a' >>= (fun a -> 
            b' >>= (fun b -> 
            return (a,b))))))
  in
  {delay = f (delay a) (delay b);
   await = f (await a) (await b)}

let delay_exp =
  let delay_exp fn = 
    fn.unpack {handler = fun closure ->
		(delay closure.op closure.env) >>= (fun env' ->
		return (env' >>= (fun env -> 
		return (build_exp {env = env; hom = closure.hom; op = closure.op}))))
	      }
  in 
  let await_exp fn = 
    fn.unpack {handler = fun closure ->
		(await closure.op closure.env) >>= (fun env' ->
		return (env' >>= (fun env -> 
		return (build_exp {closure with env = env}))))
	      }
  in {delay = delay_exp;
      await = await_exp}

let delay_sum (a : 'a delay) (b : 'b delay) =
  let f aop bop = function
    | Inl a -> (aop a) >>= (fun a' -> 
               return (a' >>= fun a -> return (Inl a)))
    | Inr b -> (bop b) >>= (fun b' -> 
               return (b' >>= fun b -> return (Inr b)))
  in
  {delay = f (delay a) (delay b);
   await = f (await a) (await b)}

let delay_next adelay =
  {delay = (fun a' -> return (a' >>= adelay.delay));
   await = (fun t -> return (return t))}

let delay_stream a =
  let f op xs = 
    (read xs) >>= (fun a -> 
    (op a) >>= (fun a' -> 
    return ((newref a') >>= (fun r -> 
            (cell (clock >>= (fun () -> 
                   (get r) >>= (fun a' ->
                   (read xs) >>= (fun x -> 
                   (op x) >>= (fun x' -> 
                   (set r x') >>= (fun () -> 
                   a'))))))) >>= (fun c ->
            (register "delay_stream" c) >>= (fun () -> 
            return c))))))
  in
  {delay = f (delay a);
   await = f (await a)}

let constant adelay a =
  (newref (return a)) >>= (fun r ->
  (cell (clock >>= (fun () -> 
         (get r) >>= (fun a' -> 
         a' >>= (fun a ->
         (delay adelay a) >>= (fun a' -> 
         (set r a') >>= (fun () -> 
         return a))))))) >>= (fun c -> 
  (register "constant" c) >>= (fun () -> 
  return c)))


let next_one = {
  into = (fun thunk -> return ());
  from = (fun () -> return (return ()))
}

let next_prod =
  let unziphalf dab =
    let da = dab >>= (fun (a,b) -> return a) in
    let db = dab >>= (fun (a,b) -> return b) in
    return (da, db) in
  let ziphalf (da,db) =
    return (da >>= (fun a -> 
            db >>= (fun b -> 
            return (a,b)))) in
  {into = unziphalf; from = ziphalf}

let map_next f da = return (da >>= f)
let map_next_iso iso = {into = map_next iso.into;
			from = map_next iso.from}


(* Function types *)

let eval (f,a) = f.unpack {handler = fun closure -> closure.hom (closure.env, a)}

let curry aop f (env : 'a) =
  return (build_exp {env = env; hom = f; op = aop})

let next_exp =
  let next_exp df = 
    return (build_exp {env = (); op = delay_one;
		       hom = fun ((), da) ->
			 return (df >>= (fun f -> 
                                 da >>= (fun a ->
				 eval (f,a))))})
  in
  let exp_next f =
    f.unpack {handler = fun closure ->
    (await closure.op closure.env) >>= (fun env' -> 
    return (env' >>= (fun env ->
            return (build_exp {env = env;
			       op = closure.op;
			       hom = fun (e,b) ->
				 closure.hom(e, return b) >>= (fun thunk ->
				 thunk)}))))}
  in
  {into = next_exp; from = exp_next}

let map_exp f h g =
  g.unpack {handler = fun closure ->
  return (build_exp {env = closure.env; op = closure.op;
		     hom = fun (env, a) ->
		       (f a) >>= (fun b ->
		       (closure.hom (env, b)) >>= (fun c ->
		       h c))})}

let map_exp_iso h k =
  {into = map_exp h.into k.into;
   from = map_exp h.from k.from}

(* Unit type *)

let one _ = return ()

(* Product type *)

let fst (a,b) = return a
let snd (a,b) = return b

let pair f g a = 
  (f a) >>= (fun b -> 
  (g a) >>= (fun c -> 
   return (b,c)))

let map_prod f g (a,b) =
  (f a) >>= (fun c -> 
  (g b) >>= (fun d -> 
  return (c,d)))

let map_prod_iso f g =
  {into = map_prod f.into g.into;
   from = map_prod f.from g.from}


let unit = {into = (fun (a, ()) -> return a);
	    from = (fun a -> return (a, ()))}

let assoc = {
  into = (fun ((a,b), c) -> return (a, (b,c)));
  from = (fun (a, (b,c)) -> return ((a,b), c))
}

let comm = {
  into = (fun (a,b) -> return (b,a));
  from = (fun (b,a) -> return (a,b))
}
	      

(* Sum type *)

let inl x = return (Inl x)
let inr x = return (Inr x)
let case f g = function
  | Inl x -> f x
  | Inr x -> g x

let map_sum f g = function
  | Inl x -> (f x) >>= inl
  | Inr y -> (g y) >>= inr

let map_sum_iso f g =
  {into = map_sum f.into g.into;
   from = map_sum f.from g.from}

let distrib =
  let into = function
    | (a, Inl b) -> return (Inl (a,b))
    | (a, Inr b) -> return (Inr (a,b)) in
  let from = function
    | Inl(a, b) -> return (a, Inl b)
    | Inr(a, b) -> return (a, Inr b) in
  {into = into; from = from}

(* Stream type *)

let head xs = read xs
let tail xs = return (return xs)

let zip =
  let zip (xs,ys) = cell((head xs) >>= (fun x -> 
                         (head ys) >>= (fun y -> 
                         return (x,y))))
  in
  let unzip xys =
    (cell (head xys >>= (fun (x,y) -> return x))) >>= (fun xs -> 
    (cell (head xys >>= (fun (x,y) -> return y))) >>= (fun ys ->
    return (xs,ys)))
  in
  {into = zip; from = unzip}

let unfold f =
  return (build_exp {
    env = f;
    op = delay_exp;
    hom = fun (f, seed) ->
      (newref (return seed)) >>= (fun s ->
      (newref (return f))    >>= (fun r -> 
      (cell (clock >>= (fun () -> 
             (get r) >>= (fun f' -> 
             (get s) >>= (fun seed' -> 
             f'      >>= (fun f -> 
             seed'   >>= (fun seed -> 
             (eval(f, seed)) >>= (fun (b, seed') -> 
             (delay delay_exp f) >>= (fun f' -> 
             (set r f') >>= (fun () -> 
             (set s seed') >>= (fun () -> 
             return b))))))))))) >>= (fun bs -> 
     (register "unfold" bs) >>= (fun () -> 
     return bs))))
	  })


type 'a cons_state =
  | Init of 'a * 'a stream code
  | Eval of 'a stream code
  | Stream of 'a stream

let cons (h, t) =
  (newref (Init (h,t))) >>= (fun r -> 
  (cell (clock >>= (fun () -> 
            (get r) >>= (function
	    | Init(h, t) -> (set r (Eval t)) >>= (fun () -> 
                            return h)
	    | Eval t -> t >>= (fun xs -> 
                        (set r (Stream xs)) >>= (fun () -> 
                        head xs))
	    | Stream xs -> head xs)))) >>= (fun c -> 
  (register "cons" c) >>= (fun () -> 
  return c)))

let map_stream f xs = cell ((head xs) >>= f)

let map_stream_iso f =
  {from = map_stream f.from;
   into = map_stream f.into}
  
let stream_strength (f, xs) =
  (constant delay_exp f) >>= (fun fs ->
  (cell ((head fs) >>= (fun f -> 
         (head xs) >>= (fun x -> 
         eval(f, x))))))
            

(* Fixed Points *)

let fix_stream bdelay (f : (('a, 'b stream next) prod, 'b stream) hom) : ('a, 'b stream) hom = fun (a : 'a) ->
  (newref None) >>= (fun (r : 'b next option ref) ->
  (cell (clock >>= (fun () -> 
         get r))) >>= (fun (preinput : 'b next option stream) -> 
  (return (cell (clock >>= (fun () -> 
          (head preinput) >>= (function
             None -> assert false
          | Some (v : 'b next) -> v))))) >>= (fun (input : 'b stream next) -> 
  (f (a,input)) >>= (fun preoutput ->
  (cell (clock >>= (fun () -> 
        (read preinput) >>= (fun _ -> 
        (read preoutput) >>= (fun b ->
        (delay bdelay b) >>= (fun b' -> 
        (set r (Some b')) >>= (fun () ->
        return b))))))) >>= (fun c -> 
  (register "fix_stream" c) >>= (fun () -> 
  return c))))))

let fix_fun f = fun a ->
  let h d = f(a,d) in
  let rec fix f = f (fun x -> fix f x) in
  fix h 

(*
  (newref None) >>= (fun r ->
  let d = (get r) >>= (function
			 | None -> assert false
			 | Some thunk -> thunk) in
  (f (a, d)) >>= (fun g ->
  (delay delay_exp g) >>= (fun g' -> 
  (set r (Some g')) >>= (fun () -> 
  return g))))
*)

let embed x = fun _ -> return x 
  
let one_discrete = {into = id; from = id}
let pair_discrete = {into = id; from = id}
let exp_discrete f =
  return (build_exp {env = (); op = delay_one;
		     hom = fun ((), x) -> return (f x)})

let run (hom : (one, 'a discrete stream) hom) : (unit -> 'a) =
  let clock = Dataflow.newcell (Dataflow.Expr.return ()) in
  let updates = ref [] in
  let result = Dataflow.eval (hom () (clock, updates)) in
  let step () =
    begin
      let v = Dataflow.eval (Dataflow.Expr.read result) in
      (List.iter Dataflow.eval !updates);
      Dataflow.update clock (Dataflow.Expr.return ());
      v
    end
  in step

(* GUI stuff *)

type window = GObj.widget
type parent = window -> unit 
type 'a gui = (parent, 'a) exp 

let fix_guistream bdelay (f : (('a, 'b stream next) prod, 'b stream gui) hom) : ('a, 'b stream gui) hom = fun (a : 'a) ->
  return
    (build_exp {
       env = (); op = delay_one;
       hom = (fun ((), (parent : parent)) -> 
        (newref None) >>= (fun (r : 'b next option ref) ->
        (cell (clock >>= (fun () -> 
               get r))) >>= (fun (preinput : 'b next option stream) -> 
        (return (cell (clock >>= (fun () -> 
                       (head preinput) >>= (fun (v' : 'b next option) ->
                       match v' with
                         None -> assert false
                       | Some (v : 'b next) -> v))))) >>= (fun (input : 'b stream next) -> 
        (f (a,input)) >>= (fun (prepreoutput : 'b stream gui) ->
        (eval(prepreoutput, parent)) >>= (fun (preoutput : 'b stream) -> 
        (cell (clock >>= (fun () -> 
              (read preinput) >>= (fun _ -> 
              (read preoutput) >>= (fun b ->
              (delay bdelay b) >>= (fun b' -> 
              (set r (Some b')) >>= (fun () ->
              return b))))))) >>= (fun c -> 
        (register "fix_guistream" c) >>= (fun () -> 
        return c))))))))
     })

let label justify = 
  fun (labels : string discrete stream) ->
    return
      (build_exp {
	 env = (); op = delay_one;
	 hom = (fun ((), (parent : parent)) -> 
          local (fun () -> GMisc.label ~justify ~packing:parent ()) >>= (fun w ->
          cell((head labels) >>= (fun (msg : string) -> 
               local (fun () -> w#set_text msg))) >>= (fun (setlabel : one stream) -> 
          register "label" setlabel >>= (fun () ->
          return ()))))
       })
								     
let button (labels : string discrete stream) =
  return 
    (build_exp {
       env = (); op = delay_one;
       hom = (fun ((), (parent : window -> unit)) -> 
        local (fun () -> GButton.button ~packing:parent ()) >>= (fun (b : GButton.button) ->
        get_clock >>= (fun clock -> 
        cell (return false) >>= (fun bsig ->
        local (fun () -> b#connect#pressed
                 ~callback:(fun () ->
        			Dataflow.update bsig (Dataflow.Expr.return true);
        			Dataflow.update clock (Dataflow.Expr.return ()))) >>= (fun _ -> 
        local (fun () -> b#connect#released
                 ~callback:(fun () ->
        			Dataflow.update bsig (Dataflow.Expr.return false);
        			Dataflow.update clock (Dataflow.Expr.return ()))) >>= (fun _ -> 
        cell (head labels >>= (fun msg -> 
              local (fun () -> b#set_label msg))) >>= (fun setlabel -> 
        register "button" setlabel >>= (fun () ->
        return bsig))))))))
     })

let checkbox labels =
  return
    (build_exp {
       env = (); op = delay_one;
       hom = (fun ((), parent) -> 
        local (fun () -> GButton.check_button ~packing:parent ()) >>= (fun b ->
        get_clock >>= (fun clock -> 
        cell (return false) >>= (fun bsig ->
        local (fun () -> b#connect#toggled
               ~callback:(fun () ->
    			let b = Dataflow.eval (Dataflow.Expr.read bsig) in 
    			Dataflow.update bsig  (Dataflow.Expr.return (not b));
    		        Dataflow.update clock (Dataflow.Expr.return ()))) >>= (fun _ ->
        cell (head labels >>= (fun msg -> 
              local (fun () -> b#set_label msg))) >>= (fun setlabel -> 
        register "checkbox" setlabel >>= (fun () ->
        return bsig)))))))
     })

let stack orientation homogeneity direction expand : ('a gui, 'a gui) hom =
  let homogeneous = match homogeneity with
    | `HOMOGENEOUS -> true
    | `INHOMOGENEOUS -> false
  in
  let expand = match expand with
    | `EXPAND -> true
    | `NOEXPAND -> false
  in
  let direction = match direction with
    | `START -> `START
    | `FINISH -> `END
  in 
  fun (gui : 'a gui) ->
    return 
      (build_exp {
         env = (); op = delay_one;
         hom = (fun ((), (parent : window -> unit)) ->
          local (fun () -> GPack.box orientation ~homogeneous () ~packing:parent) >>= (fun box ->
          eval(gui, (fun w -> box#pack ~expand ~from:direction w))))
       })

let delay_gui = delay_exp

let bind f = curry delay_discrete (compose (pair (compose eval f) snd) eval)

let return = fun x -> (curry delay_discrete fst) x

let strength : (('a, 'b gui) prod, ('a, 'b) prod gui) hom = 
  fun x -> (curry delay_discrete (compose (pair (compose fst fst) (pair (compose fst snd) snd))
				    (pair fst (compose snd eval)))) x 


let guirun (f : (one, one gui) hom) : unit = 
  let clock = Dataflow.newcell (Dataflow.Expr.return ()) in
  let updates = ref [] in
  let gui = Dataflow.eval (f () (clock, updates)) in
  let w = GWindow.window ~border_width:10 () in
  let _ = w#connect#destroy ~callback:GMain.Main.quit in
  let root = GPack.vbox ~packing:w#add () in 
  let () = Dataflow.eval ((eval(gui, root#add)) (clock, updates)) in
  let _ = GMain.Idle.add
	    (fun () -> List.iter Dataflow.eval (!updates); true) in
(*
  let _ = GMain.Timeout.add
	  ~ms:10
	  ~callback:(fun () -> Dataflow.update clock (Dataflow.Expr.return ()); true) in
*)
  let _ = w#show () in 
    GMain.Main.main ()

