#!/usr/bin/env perl
# Test example:
# ./commit-update.pl /usr/groups/wwwsvn/repositories/vh-cl 170 /home/mgk25/public_html/cl-preview/html -s trunk/html/

# this software requires Perl 5.14 or newer
use 5.014;  # implies use strict;
use POSIX qw(strftime);

my $usage =
    "Subversion post-commit script: updates files in a given\n".
    "working directory and calls ucampas where appropriate\n\n".
    "usage: $0 REPOS REVNUM working_dir [options]\n".
    "       $0 -n 0 working_dir [options]\n\n".
    "options:\n\n".
    "  -s strip_prefix    remove this prefix from pathname in repository\n".
    "                     before appending the rest to working directory\n".
    "  -m match_regexp    process only files whose path in the repository\n".
    "                     (after -s was applied) matches this regular expr.\n".
    "  -q                 quiet mode: output only error messages\n".
    "  -l logfile         write command output to this logfile\n".
    "                     (relative paths are relative to working_dir)\n".
    "  -e addr{,addr}     send command output to these email addresses ...\n".
    "  -c addr{,addr}     ... and these Cc: addresses ...\n".
    "  -f addr            ... using this From: address\n".
    "  -r rootdir         ucampas top-level directory where background\n".
    "                     update should start (if not in working_dir)\n".
    "                     (relative paths are relative to working_dir)\n".
    "  -u ucampas_path    path of ucampas executable (if not on \$PATH)\n".
    "  -M                 execute 'make -f .makefile' or 'make -f Makefile'\n".
    "                     in any changed directory that contains one\n\n".
    "Arguments '-n 0' instead of 'REPOS REVNUM' suppress processing a commit\n".
    "and just start the final 'ucampas -r &' background update.\n";

my $start_time = time;
my @invocation = @ARGV;
my $repos  = shift @ARGV;
my $revnum = shift @ARGV;
my $wdir   = shift @ARGV;
my $pattern;
my $prefix;
my $callmake;

my $maxargs = 20;
my $ucampas = "ucampas";
my $errors = 0;
my $quiet = '';

my @log;
my $logfile;
my $emails;
my $cc;
my $from = 'webmaster';
my $rdir;

umask 002;

# Quote a string for the shell (surround with "..." and
# escape $`"\ with backslash)
sub shellquote($) {
    my ($s) = @_;

    $s =~ s/([\$\`\"\\])/\\$1/g;
    $s = '"' . $s . '"';

    return $s;
}

sub cmd {
    my ($cmd) = @_;
    push @log, "$cmd\n" unless $quiet;
    push @log, `$cmd 2>&1`;
    if ($?) {
	push @log, "Failed command: $cmd\n";
	$errors++;
    }
}

# Prefix relative paths in @paths with $wdir, separated by /,
# avoiding double and trailing slashes, but leave alone absolute
# paths in @paths starting with /
sub prefix_path {
    my ($wdir, @paths) = @_;
    $wdir =~ s|(?<=.)/+\z||; # remove trailing slashes
    my @r = map {
	s|(?<=.)/+\z||;      # remove trailing slashes
	if (m|^/|) {
	    ($_);            # preserve absolute path
	} elsif ($_ eq '' || $_ eq '.' || $_ eq './' ) {
	    ($wdir);         # empty suffix
	} elsif ($wdir =~ m|^/?\z|) {
	    ("$wdir$_");     # no slash required
	} else {
	    ("$wdir/$_");
	}
    } @paths;
    return @r if wantarray;
    die unless @paths == 1;
    return $r[0];
}

# remove prefix $prefix from paths in @paths
sub strip_prefix {
    my ($prefix, @paths) = @_;

    $prefix .= '/' unless $prefix =~ m|/\z|;

    return map {
	(substr($_, 0, length($prefix)) eq $prefix) ?
	    substr($_, length($prefix)) : ();
    } @paths;
}

sub command {
    my ($prog, @args) = @_;
    my @skip;
    my @process;

    # deal only with owned or not (yet) existing files
    foreach my $f (@args) {
	if (-o $f || ! -e $f) {
	    push @process, shellquote($f);
	} else {
	    push @skip, shellquote($f);
	}
    }
    if (@skip) {
	push @log, "Skipping files not owned ($prog):\n";
	cmd("ls -ld ". join(' ', splice(@skip, 0, $maxargs))) while (@skip);
    }
    while (@process) {
	cmd("$prog ". join(' ', splice(@process, 0, $maxargs)));
    }
}


# Trigger a rebuild of the entire main site as a background job
# to cover changes to navigation structure, sitemap, etc.
# Make sure that not more than one of these jobs is running concurrently,
# by terminating an already running one before starting the new one.
sub background_rebuild
{
    my $rdir = shift;
    my $qrdir = shellquote($rdir);
    if (open(PID, '<', "$rdir/.backgroundpid")) {
	kill TERM => <PID>;
	close PID;
	sleep 2;  # give the background rm a chance to finish first
	unlink "$rdir/.backgroundpid";  # just to be save
    }
    unlink "$rdir/.backgroundlog";
    `{ nice sh -c 'echo \$\$ >$qrdir/.backgroundpid ; exec $ucampas -r $qrdir' ; rm -f $qrdir/.backgroundpid ; } </dev/null >$qrdir/.backgroundlog 2>&1 &`;
}

eval {

    die $usage unless ($repos && $wdir);

    $rdir = $wdir;
    while (@ARGV) {
	my $arg = shift @ARGV;
	if ($arg =~ /^-m$/) {
	    $pattern = shift @ARGV;
	} elsif ($arg =~ /^-M$/) {
	    $callmake = 1;
	} elsif ($arg =~ /^-q$/) {
	    $quiet = ' -q';
	} elsif ($arg =~ /^-s$/) {
	    $prefix = shift @ARGV;
	} elsif ($arg =~ /^-l$/) {
	    $logfile = shift @ARGV;
	    $logfile = prefix_path($wdir, $logfile);
	} elsif ($arg =~ /^-e$/) {
	    $emails = shift @ARGV;
	} elsif ($arg =~ /^-c$/) {
	    $cc = shift @ARGV;
	} elsif ($arg =~ /^-f$/) {
	    $from = shift @ARGV;
	} elsif ($arg =~ /^-r$/) {
	    $rdir = shift @ARGV;
	    $rdir = prefix_path($wdir, $rdir);
	} elsif ($arg =~ /^-u$/) {
	    $ucampas = shift @ARGV;
	} else {
	    die "unexpected option '$arg'\n" . $usage;
	}
    }

    # sanity checks
    die("Working directory '$wdir' not readable\n")
	unless -r $wdir;
    die("Ucampas background-run top-level directory '$rdir' not writeable:\n".
	`ls -ld $rdir/`)
	unless -w $rdir;

    return if $repos eq '-n';

    # more sanity checks
    die("Repository '$repos' not readable\n")
	unless -r $repos;
    die("Revision '$revnum' is not an integer\n")
	unless $revnum =~ /^\d+$/;

    push @log, `svn log -v file://$repos --revision $revnum` unless $quiet;

    my $cmd = "svnlook dirs-changed $repos --revision $revnum";
    $_ = `$cmd`; $? and die("Failed command: $cmd\n");
    chomp;
    my @changed_dirs = split /\n/;
    # avoid '/' being mis-interpreted later as absolute path
    for my $d (@changed_dirs) { $d = './' if $d eq '/'; }

    $cmd = "svnlook changed $repos --revision $revnum";
    $_ = `$cmd`; $? and die("Failed command: $cmd\n");
    chomp;
    my @updated_files = split /\n/;
    for my $d (@updated_files) { $d =~ s|^(...) /$|$1 ./|; }

    my @deleted_files = grep /^D.. .*[^\/]$/, @updated_files;
    my @deleted_dirs  = grep /^D.. .*\/$/,    @updated_files;
    my @changed_files = grep /^[UAG].. /,     @updated_files;
    @deleted_files = grep { s/^... //; } @deleted_files;
    @deleted_dirs  = grep { s/^... //; } @deleted_dirs;
    @changed_files = grep { s/^... //; } @changed_files;

    if (defined $prefix) {
	@changed_dirs  = strip_prefix($prefix, @changed_dirs);
	@changed_files = strip_prefix($prefix, @changed_files);
	@deleted_dirs  = strip_prefix($prefix, @deleted_dirs);
	@deleted_files = strip_prefix($prefix, @deleted_files);
    }

    if (defined $pattern) {
	@changed_dirs  = grep { /$pattern/ } @changed_dirs;
	@changed_files = grep { /$pattern/ } @changed_files;
	@deleted_dirs  = grep { /$pattern/ } @deleted_dirs;
	@deleted_files = grep { /$pattern/ } @deleted_files;
    }

    return unless @changed_dirs;  # nothing left to do

    @changed_dirs  = prefix_path($wdir, @changed_dirs);
    @changed_files = prefix_path($wdir, @changed_files);
    @deleted_dirs  = prefix_path($wdir, @deleted_dirs);
    @deleted_files = prefix_path($wdir, @deleted_files);

    # remove files whose ucampas sources are about to be deleted
    command('rm -f', grep { s/-b(\.(?:html?|php))$/$1/i && -e } @deleted_files);
    # same for all *-b.html generated *.html files in
    # about-to-be-deleted directories
    foreach my $deleted_dir (@deleted_dirs) {
	cmd("find '$deleted_dir' -name .svn -prune -o -iregex '.*-b\\.\\(html?\\|php\\)\\|.*~' -print0 | perl -0 -pe 's/-b(\\.(?:html?|php)\\0)\$/\\1/i' | xargs --no-run-if-empty -0 rm -f");
    }

    # check for administrative locks in the working directory,
    # to avoid misleading error message in case the commit happened
    # from within the same working directory and its locks are still there
    $cmd = 'svn status --ignore-externals ' .
	join(' ', map { shellquote($_) } @changed_dirs);
    $_ = `$cmd`; $? and die("Failed command: $cmd\n");
    chomp;
    my @status = split /\n/;
    if (grep { /^..L/ } @status) {
	push @log, "Working directory is locked, skipping svn update.\n";
	push @log, $_;
	return;
    }

    # update any affected directories
    # (earlier attempts with "svn update -N" broke way too much)
    command("svn update --ignore-externals" . $quiet, @changed_dirs);

    # call ucampas on any added, updated or merged *-b.html files
    my @changed_b_files = grep { /-b\.(?:html?|php)$/i } @changed_files;
    command($ucampas . $quiet, @changed_b_files);
    # ensure that ucampas does get called if a uconfig.txt file was changed,
    # such that problems there are reported back to the committer instantly
    my @changed_uconfig_files = grep { /^(?:.*\/)?uconfig.txt$/ } @changed_files;
    if (@changed_uconfig_files && !@changed_b_files) {
	# rebuild root page to test uconfig.txt changes
	# (todo: rebuild instead index of directory where the
	# changed uconfig.txt file resides)
	command($ucampas . $quiet, "$rdir/") if $rdir;
    }

    if ($callmake) {
	foreach my $makedir (@changed_dirs) {
	    if (-r "$makedir/.makefile") {
		command('make' . ($quiet ? ' -s' : '') . ' -f .makefile -C',
			$makedir);
	    } elsif (-r "$makedir/Makefile") {
		command('make' . ($quiet ? ' -s' : '') . ' -f Makefile -C',
			$makedir);
	    }
	}
    }

};
if ($@) {
    push @log, $@;
    $errors++;
}

# email, output and file log of commands executed
my $log = join('', @log);
if (defined $emails && $errors) {
    my $message =
	  "From: $from\n" .
	  "To: $emails\n" .
	  ($cc ? "Cc: $cc\n" : '') .
	  "Subject: problem with svn-commit r$revnum\n" .
	  "X-script: $0\n" .
	  "X-repository: $repos\n" .
	  "X-wdir: $wdir\n" .
	  "Mime-version: 1.0\nContent-type: text/plain; charset=UTF-8\n\n" .
	  "This is an automatically generated commit-update " .
	  "error message.\n" .
	  "When processing svn commit $revnum, " .
	  "this problem occurred:\n\n" . $log;
    my $sendmail = "/usr/sbin/sendmail -t -B 8BITMIME -i -f $from";
    if (open(MAIL, '|-', $sendmail)) {
	print MAIL $message;
	close MAIL
	    or $log .= "Failed closing $sendmail: $!\n$message";
    } else {
	$log .= "Failed calling $sendmail: $!\n";
    }
}
print STDERR $log;
if (defined $logfile) {
    my $finish_time = time;
    open(LOG, '>>', $logfile);
    print LOG
	(strftime("\nStarted %Y-%m-%d %H:%M:%S\n",
		  localtime($start_time)) .
	 join(' ', $0, @invocation) . "\n" .
	 $log .
	 strftime("Finished %Y-%m-%d %H:%M:%S" , localtime($finish_time)) .
	 sprintf(" (%d s, %d errors)\n", $finish_time - $start_time, $errors));
    close LOG;
}

# Finally, trigger a rebuild of the entire main site as a background job
background_rebuild($rdir) if $rdir;

exit $errors > 0;
