/*
 * StirMark -- Experimental watermark resilience testkit
 *
 * Markus Kuhn <mkuhn@acm.org>, University of Cambridge
 *
 * StirMark applies a number of minor almost invisible distortions to
 * an image in order to test whether some watermark detector can still
 * find the embedded signal after this processing. The distortion simulate
 * a resampling process, i.e. they include a minor geometric transform,
 * linear interpolation, as well as slightly non-identical transfer
 * functions. The distortions are adjusted to be not recognizeable
 * for typical photographic source images by human viewers.
 *
 * (c) 1997 Markus Kuhn
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>

#ifndef PI
#define PI        3.14159265358979323846
#endif

#define MAX_DEPTH 3             /* maximum number of bytes per pixel */
#define SINC_RESOLUTION 20      /* number of table points per unity distance */

int cutoff = 6;                 /* sinc(x) = 0 for all x > cutoff */
double *sinc_point;             /* array of values for sinc interpolation,
				 * size = SINC_RESOLUTION * cutoff + 1 */


void *checkedmalloc(size_t n)
{
  void *p;
  
  if ((p = malloc(n)) == NULL) {
    fprintf(stderr, "Sorry, not enough memory available!\n");
    exit(1);
  }
  
  return p;
}


/* random number generator, return a random number in the range [0,1) */

double rng(void)
{
  return (double)rand()/RAND_MAX;   /* there are certainly better ones ... */
}


/* graphic file handling routines */

/* skip whitespace and comments in PGM/PPM headers */
void skip_white(FILE *f)
{
  int c;
  
  do {
    while (isspace(c = getc(f)));
    if (c == '#')
      while ((c = getc(f)) != '\n' && c != EOF);
    else {
      ungetc(c, f);
      return;
    }
  } while (c != EOF);
  
  return;
}


void read_pnm(FILE *fin, unsigned char **img,
	      int *xsize, int *ysize, int *depth, int *max)
{
  char magic[10];
  int i, v;

  fgets(magic, 10, fin);
  if (magic[0] != 'P' || !isdigit(magic[1]) || magic[2] != '\n') {
    fprintf(stderr, "Unsupported input file type!\n");
    exit(1);
  }
  skip_white(fin);
  fscanf(fin, "%d", xsize);
  skip_white(fin);
  fscanf(fin, "%d", ysize);
  skip_white(fin);
  fscanf(fin, "%d", max);
  getc(fin);
  if (*max > 255 || *max <= 0 || *xsize <= 1 || *ysize <= 1) {
    fprintf(stderr, "Unsupported value range!\n");
    exit(1);
  }

  switch (magic[1]) {
  case '2': /* PGM ASCII */
  case '5': /* PGM binary */
    *depth = 1; /* up to 8 bit/pixel */
    break;
  case '3': /* PPM ASCII */
  case '6': /* PPM binary */
    *depth = 3; /* up to 24 bit/pixel */
    break;
  default:
    fprintf(stderr, "Unsupported input file type 'P%c'!\n", magic[1]);
    exit(1);
  }
  
  *img = (unsigned char *) checkedmalloc(sizeof(unsigned char) *
					*xsize * *ysize * *depth);
  
  switch (magic[1]) {
  case '2': /* PGM ASCII */
  case '3': /* PPM ASCII */
    for (i = 0; i < *xsize * *ysize * *depth; i++) {
      skip_white(fin);
      fscanf(fin, "%d", &v);
      if (v < 0 || v > *max) {
	fprintf(stderr, "Out of range value!\n");
	exit(1);
      }
      (*img)[i] = v;
    }
    break;
  case '5': /* PGM binary */
  case '6': /* PPM binary */
    fread(*img, *xsize * *depth, *ysize, fin);
    break;
  }

  if (ferror(fin)) {
    perror("Error occured while reading input file");
    exit(1);
  }
  if (feof(fin)) {
    fprintf(stderr, "Unexpected end of input file!\n");
    exit(1);
  }

}


/* sin(PI*x)/(PI*x) interpolation code */

void init_sinc(void)
{
  int i, j;

  sinc_point = (double *) checkedmalloc(sizeof(double) *
					(cutoff * SINC_RESOLUTION + 1));

  sinc_point[0] = 1.0;
  for (j = 1; j <= cutoff * SINC_RESOLUTION; j++)
    sinc_point[j] = 0.0;

  /*
   * Add sinc values outside the cutoff distance to the values
   * inside the cutoff distance to preserve overall energy. For
   * numerical stability, start with the small outside values.
   */
  for (i = SINC_RESOLUTION * cutoff * 100; i > 0;) {
    for (j = 0; j < cutoff * SINC_RESOLUTION; j++, i--)
      sinc_point[j] += sin(PI * i / (double) SINC_RESOLUTION) /
	(PI * i / (double) SINC_RESOLUTION);
    for (j = cutoff * SINC_RESOLUTION; j > 0; j--, i--)
      sinc_point[j] += sin(PI * i / (double) SINC_RESOLUTION) /
	(PI * i / (double) SINC_RESOLUTION);
  }

  return;
}


double sinc(double x)
{
  int b;

  x = fabs(x) * SINC_RESOLUTION;
  b = x;
  if (b >= cutoff * SINC_RESOLUTION)
    return 0;
  /* linear interpolation */
  return (1 - (x - b)) * sinc_point[b] + (x - b) * sinc_point[b+1];
}


int main(int argc, char **argv)
{
  char version[] = "StirMark 1.0 -- (c) 1997 Markus Kuhn";
  char usage[] = "%s\n\n%s [options] [<input file> [<output file>]]\n"
    "\t-i<float>\tmaximum distance a corner can move inwards (2%%)\n"
    "\t-o<float>\tmaximum distance a corner can move outwards (0.7)\n"
    "\t-d<float>\tmaximum variation of a pixel value (1.5)\n"
    "\t-n<int>\t\tuse Nyquest interpolation up to specified distance (6)\n"
    "\t-r<float>\trelative modification of Nyquist filter threshhold (1.0)\n"
    "\t-l<int>\t\tuse faster linear interpolation\n"
    "\t-s<int>\t\trandom number generator seed\n\n";

  FILE *fin = stdin, *fout = stdout;
  unsigned char *img;
  int xsize, ysize, depth, max;
  double xmax, ymax;
  int i, j, k, vb, l, m;
  int winsize_x, winsize_y;
  double v[MAX_DEPTH], w;
  double inwards = 2, outwards = 0.7, deviation = 1.5;
  int relative_inwards = 1, relative_outwards = 0;
  double rolloff_x = 1.0, rolloff_y = 1.0;
  int highpass_x = 0, highpass_y = 0;
  double ulx, uly, llx, lly, urx, ury, lrx, lry;   /* new corner offsets */
  double x, y;                                     /* sample coordinates */
  int pulx, puly;                                  /* corner pix in patch */
  int px, py;                                      /* neighbor pix for conv */
  int nyquist = 1;                                 /* interpolation type */
  int perr;                                        /* number of error patch */
#define ERRSTEP 16
  double err[256/ERRSTEP+1][MAX_DEPTH];


  /* read command line arguments */
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '/' && argv[i][1] == '?' && argv[i][2] == 0) {
      fprintf(stderr, usage, version, argv[0]);
      exit(1);
    } else if (argv[i][0] == '-')
      for (j = 1; j > 0 && argv[i][j] != 0; j++)
	switch (argv[i][j]) {
	case 'i':
	  inwards = atof(argv[i] + j + 1);
	  relative_inwards = argv[i][strlen(argv[i])-1] == '%';
	  j = -1;
	  break;
	case 'o':
	  outwards = atof(argv[i] + j + 1);
	  relative_outwards = argv[i][strlen(argv[i])-1] == '%';
	  j = -1;
	  break;
	case 'd':
	  deviation = atof(argv[i] + j + 1);
	  j = -1;
	  break;
	case 'r':
	  if (argv[i][j+1] == 'x')
	    rolloff_x = atof(argv[i] + j + 2);
	  else if (argv[i][j+1] == 'y')
	    rolloff_y = atof(argv[i] + j + 2);
	  else
	    rolloff_x = rolloff_y = atof(argv[i] + j + 1);
	  j = -1;
	  break;
	case 's':
	  srand(atoi(argv[i] + j + 1));
	  j = -1;
	  break;
	case 'n':
	  /* use true Nyquist sin(x)/x interpolation */
	  nyquist = 1;
	  if (isdigit(argv[i][j+1])) {
	    cutoff = atoi(argv[i] + j + 1);
	    j = -1;
	  }
	  break;
	case 'l':
	  /* use cheap bilinear interpolation */
	  nyquist = 0;
	  break;
	default:
	  fprintf(stderr, usage, version, argv[0]);
	  exit(1);
	}
    else if (fin == stdin) {
      fin = fopen(argv[i], "rb");
      if (fin == NULL) {
        fprintf(stderr, "Can't open input file '%s", argv[i]);
        perror("'");
        exit(1);
      }
    } else if (fout == stdout) {
      fout = fopen(argv[i], "wb");
      if (fout == NULL) {
        fprintf(stderr, "Can't open output file '%s", argv[i]);
        perror("'");
        exit(1);
      }
    } else {
      fprintf(stderr, usage, version, argv[0]);
      exit(1);
    }
  }

  if (rolloff_x < 0) rolloff_x = -rolloff_x, highpass_x = 1;
  if (rolloff_y < 0) rolloff_y = -rolloff_y, highpass_y = 1;

  read_pnm(fin, &img, &xsize, &ysize, &depth, &max);
  if (relative_inwards)
    inwards = (xsize < ysize ? xsize : ysize) * inwards / 100.0;
  if (relative_outwards)
    outwards = (xsize < ysize ? xsize : ysize) * outwards / 100.0;

  init_sinc();

  /* set corner offsets */
  ulx = rng()*(inwards+outwards)-outwards;
  uly = -rng()*(inwards+outwards)+outwards;
  llx = rng()*(inwards+outwards)-outwards;
  lly = rng()*(inwards+outwards)-outwards;
  urx = -rng()*(inwards+outwards)+outwards;
  ury = -rng()*(inwards+outwards)+outwards;
  lrx = -rng()*(inwards+outwards)+outwards;
  lry = rng()*(inwards+outwards)-outwards;
  /* set transfer function error offsets */
  for (i = 0; i <= 256/ERRSTEP; i++)
    for (k = 0; k < depth; k++)
      err[i][k] = rng()*2*deviation - deviation;

  fprintf(fout, "P%d\n%d %d\n%d\n", depth == 1 ? 5 : 6, xsize, ysize, max);

  xmax = xsize - 1;
  ymax = ysize - 1;

  for (j = 0; j < ysize; j++) {
    for (i = 0; i < xsize; i++) {
      /* Bilinear interpolation:
       * (UL*(1-j/ymax)+LL*(j/ymax)) * (1-i/xmax) +
       * (UR*(1-j/ymax)+LR*(j/ymax)) * (i/xmax)
       */
      /* coordinates of the new sample point in the old image */
      x =
	(ulx*(1-j/ymax)+llx*(j/ymax)) * (1-i/xmax) +
	((urx+xmax)*(1-j/ymax)+(lrx+xmax)*(j/ymax)) * (i/xmax);
      y =
	(-uly*(1-j/ymax)+(-lly+ymax)*(j/ymax)) * (1-i/xmax) +
	((-ury)*(1-j/ymax)+(-lry+ymax)*(j/ymax)) * (i/xmax);
      /* coordinates of the corners of the patch in which the new sample
       * point (x,y) is located are (pulx, puly), (pulx+1, puly),
       * (pulx, puly+1), (pulx+1, puly+1) */
      pulx = (int) x;
      puly = (int) y;
      if (pulx < 0) pulx = 0;
      if (pulx > xsize-2) pulx = xsize-2;
      if (puly < 0) puly = 0;
      if (puly > ysize-2) puly = ysize-2;

      if (nyquist) {
	/* convolute with 2D sinc to get a sample value v */
	winsize_x = cutoff/rolloff_x + 0.5;
	winsize_y = cutoff/rolloff_y + 0.5;
	v[0] = v[1] = v[2] = 0;
	for (l = -winsize_x+1; l <= winsize_x; l++) {
	  px = pulx + l;
	  /* mirror at borders */
	  while (px < 0 || px >= xsize) {
	    if (px < 0) px = -px;
	    if (px >= xsize) px = 2 * (xsize - 1) - px;
	  }
	  for (m = -winsize_y+1; m <= winsize_y; m++) {
	    py = puly + m;
	    /* mirror at borders */
	    while (py < 0 || py >= ysize) {
	      if (py < 0) py = -py;
	      if (py >= ysize) py = 2 * (ysize - 1) - py;
	    }
	    if (highpass_x)
	      w = sinc(pulx + l - x) - 
		sinc((pulx + l - x) * rolloff_x) * rolloff_x;
	    else
	      w = sinc((pulx + l - x) * rolloff_x) * rolloff_x;
	    if (highpass_y)
	      w *= sinc(puly + m - y) -
		sinc((puly + m - y) * rolloff_y) * rolloff_y;
	    else
	      w *= sinc((puly + m - y) * rolloff_y) * rolloff_y;
	    for (k = 0; k < depth; k++)
	      v[k] += img[k + depth * (px + xsize * py)] * w;
	  }
	}
      } else {
	/* bi-linear interpolation to get a sample value v */
        for (k = 0; k < depth; k++) {
	  v[k] =
	    (img[k+depth*(pulx+xsize*puly)]*(1-(y-puly)) +
	     img[k+depth*(pulx+xsize*(puly+1))]*(y-puly)) * (1-(x-pulx)) +
	    (img[k+depth*((pulx+1)+xsize*puly)]*(1-(y-puly)) +
	     img[k+depth*((pulx+1)+xsize*(puly+1))]*(y-puly)) * (x-pulx);
	}
      }
      for (k = 0; k < depth; k++) {
	/* add offset when highpass was used */
	if (highpass_x || highpass_y)
	  v[k] += max / 2.0;
	/* interpolation of transfer function offset */
	perr = v[k] / ERRSTEP;
	if (perr < 0) perr = 0;
	if (perr >= 256/ERRSTEP) perr = 256/ERRSTEP-1;
	v[k] += 
	  (1-(v[k]-perr*ERRSTEP)/ERRSTEP)*err[perr][k] +
	  ((v[k]-perr*ERRSTEP)/ERRSTEP)*err[perr+1][k];
	/* dithering to distribute quantization noise */
	vb = (int) (v[k] + rng());
	/* thresholding */
	if (vb < 0) vb = 0;
	if (vb > max) vb = max;
	/* done */
	fputc(vb, fout);
      }
    }
  }

  return 0;
}
