#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "osc.h"

#define BUFLEN	16384
#define OUTDATLEN	65536

void
main(void)
{
	struct in_addr	in_addr;
	struct hostent	*hostent;
	char	waveformbuf[BUFLEN], preamblebuf[BUFLEN], *data,
		textbuf[OUTDATLEN];
	int	i, waveformlen, preamblelen, out;

	long	p_format, p_type, p_points, p_count;
	double	p_xinc, p_xorig, p_xref, p_yinc, p_yorig, p_yref;

	hostent = gethostbyname("infocalypse.al.cl.cam.ac.uk");
	if (!hostent) {
		perror("gethostbyname");
		exit(-1);
	}

	if (hostent->h_addrtype != AF_INET) {
		fprintf(stderr, "gethostbyname returned non-inet address\n");
		exit(-1);
	}

	if (hostent->h_length != sizeof(int)) {
		fprintf(stderr, "gethostbyname did not return an address\n");
		exit(-1);
	}

	in_addr.s_addr = *(int*)*hostent->h_addr_list;

	printf("%s\n", inet_ntoa(in_addr));

	i = osc_connect(in_addr, 4500, "GPIB0::7::INSTR");
	if (i) {
		perror("osc_connect");
		exit(-1);
	}

	i = osc_write(":dig");
	if (i == -1) {
		perror("osc_write");
		exit(-1);
	}

	i = osc_write(":waveform:data?");
	if (i == -1) {
		perror("osc_read");
		exit(-1);
	}

	i = osc_read(waveformbuf, BUFLEN);
	if (i <= 0) {
		perror("osc_read");
		exit(-1);
	}
	waveformlen = i;

	i = osc_waveform_parse(waveformbuf, waveformlen, &data);
	if (i == -1) {
		fprintf(stderr, "osc_waveform_parse failed\n");
		exit(-1);
	}

	printf("Read %d bytes from oscilloscope, data is %d bytes\n",
	    waveformlen, waveformlen - (data - waveformbuf));

	i = osc_write(":waveform:preamble?");
	if (i == -1) {
		perror("osc_write");
		exit(-1);
	}

	i = osc_read(preamblebuf, BUFLEN);
	if (i <= 0) {
		perror("osc_read");
		exit(-1);
	}

	preamblelen = i;
	preamblebuf[preamblelen] = '\0';
	i = osc_parse_preamble(preamblebuf, 
			       &p_format, &p_type, &p_points, &p_count,
			       &p_xinc, &p_xorig, &p_xref,
			       &p_yinc, &p_yorig, &p_yref);
	if (i == -1) {
		fprintf(stderr, "osc_parse_preamble: unable to parse\n");
		exit(-1);
	}

	printf("Preamble: %s\n", preamblebuf);

	i = unlink("out.dat");
	/* ignore failure, just need to remove old data */

	i = open("out.dat", O_CREAT | O_WRONLY, 0666);
	if (i == -1) {
		perror("open");
		exit(-1);
	}

	out = i;

	i = osc_waveform_to_text(textbuf, OUTDATLEN, data,
				 waveformlen - (data - waveformbuf),
				 p_xinc, p_xorig, p_xref,
				 p_yinc, p_yorig, p_yref);
	if (i == -1) {
		fprintf(stderr, "osc_waveform_to_text: unable to convert\n");
		exit(-1);
	}

	printf(textbuf);

	i = write(out, textbuf, strlen(textbuf));
	if (i != strlen(textbuf)) {
		perror("write");
		exit(-1);
	}
	close(out);
}
