/*
 * Glyph test tool for X11
 *
 * (c) 1999 Markus Kuhn
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/keysym.h>

Display *display;
int screen;
Window win;
GC gc_text_bg;
Colormap colormap;

#define MAXFONTS 32
XFontStruct *font_info[MAXFONTS];
char *font_name[MAXFONTS];
GC gc_text[MAXFONTS];
int fonts;
int lineskip = 0, lines = 1;
int helper=0;
int fontnamelines = 1;

int firstchar = 0;
#define CLISTLEN 0x10000
int chars;
struct {
  unsigned short ucs;
  char *text;
} clist[CLISTLEN];

#define testglyph_bitmap_width 40
#define testglyph_bitmap_height 40
static unsigned char testglyph_bitmap_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0x01, 0x00, 0x00, 0x00, 0x03, 0x06, 0x00, 0x00,
  0xc0, 0xf0, 0x1f, 0x00, 0x00, 0x20, 0x0c, 0x38, 0x00, 0x00, 0x10, 0x03,
  0x60, 0x00, 0x00, 0x88, 0x20, 0x80, 0x00, 0x00, 0x44, 0x70, 0x00, 0x01,
  0x00, 0x24, 0xf8, 0x00, 0x03, 0x00, 0x12, 0xd8, 0x00, 0x06, 0x00, 0x12,
  0x8c, 0x01, 0x06, 0x00, 0x09, 0x8c, 0x01, 0x0c, 0x00, 0x09, 0x06, 0x03,
  0x0c, 0x00, 0x09, 0x06, 0x03, 0x0c, 0x00, 0x05, 0xff, 0x07, 0x14, 0x00,
  0x05, 0xff, 0x07, 0x14, 0x00, 0x85, 0x01, 0x0c, 0x14, 0x00, 0x85, 0x01,
  0x0c, 0x14, 0x00, 0xc5, 0x00, 0x18, 0x14, 0x00, 0xc6, 0x00, 0x18, 0x12,
  0x00, 0xf6, 0x01, 0x7c, 0x12, 0x00, 0x0c, 0x00, 0x00, 0x09, 0x00, 0x0c,
  0x00, 0x00, 0x09, 0x00, 0x0c, 0x00, 0x80, 0x08, 0x00, 0x12, 0x00, 0x40,
  0x04, 0x00, 0x31, 0x00, 0x20, 0x04, 0x80, 0xf0, 0x00, 0x18, 0x02, 0x40,
  0x78, 0x03, 0x06, 0x01, 0x20, 0x8c, 0xfc, 0x81, 0x00, 0x10, 0x06, 0x03,
  0x60, 0x00, 0x08, 0x03, 0x0c, 0x18, 0x00, 0x84, 0x01, 0xf0, 0x07, 0x00,
  0xcc, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


void redraw()
{
  XWindowAttributes attrib;
  char line[80];
  XChar2b sample[3];
  int i, j, y;
  int columnwidth;

  XGetWindowAttributes(display, win, &attrib);

  columnwidth = helper ? 45 : 18;

  y = 10 + (3 + font_info[0]->ascent + font_info[0]->descent) *
    (fontnamelines + 1);
  lines = (attrib.height - y) / lineskip;
  XFillRectangle(display, win, gc_text_bg,
                 0, 0, attrib.width, attrib.height);
  for (i = 1; i < fonts; i++)
    XDrawLine(display, win, gc_text[0],
	      60 + columnwidth * (i-1) + 3,
	      (3 + font_info[0]->ascent + font_info[0]->descent) *
	      (((i-1) % fontnamelines) + 1) + 3,
	      60 + columnwidth * (i-1) + 3,
	      (3 + font_info[0]->ascent + font_info[0]->descent) *
	      (fontnamelines + 1) - 8);
  for (i = 1; i < fonts; i++) {
    XDrawString(display, win, gc_text[0],
		60 + columnwidth * (i-1),
		(3 + font_info[0]->ascent + font_info[0]->descent) *
		(((i-1) % fontnamelines) + 1),
		font_name[i], strlen(font_name[i]));
  }

  for (j = 0; j < lines && j + firstchar <= chars-1; j++, y += lineskip) {
    sprintf(line,"U+%04X", clist[firstchar + j].ucs);
    XDrawString(display, win, gc_text[0], 10, y, line, strlen(line));
    sample[0].byte1 = 0;
    sample[0].byte2 = 'H';
    sample[1].byte1 = clist[firstchar + j].ucs >> 8;
    sample[1].byte2 = clist[firstchar + j].ucs & 0xff;
    sample[2].byte1 = 0;
    sample[2].byte2 = 'x';
    for (i = 1; i < fonts; i++) {
      XDrawString16(display, win, gc_text[i], 60 + columnwidth * (i-1), y,
		  sample + !helper, 3 - 2 * !helper);
    }
    XDrawString(display, win, gc_text[0], 70 + columnwidth * (fonts-1), y,
		clist[firstchar + j].text, strlen(clist[firstchar + j].text));
  }
}


int main(int argc, char **argv)
{
  char version[] = "glyphtest 0.3 -- (c) 1999 Markus Kuhn";
  char usage[] = "%s\n\n%s [-] {fonts}\n";
  Pixmap icon_pixmap;
  XEvent event;
  XSizeHints sizehints;
#define KEYLEN 10
  char key[KEYLEN];
  KeySym keysym;
  XComposeStatus composestatus;
  char *display_name = NULL;
  int i, j;
  char line[200];

  font_name[0] = "fixed";
  fonts = 1;

  /* parse command line */
  chars = 0;
  for (i = 1; i < argc; i++) {
    if (!strcmp(argv[i], "-")) {
      while (fgets(line, sizeof(line), stdin)) {
	for (j = strlen(line) - 1; j >= 0 && isspace(line[j]); line[j--] = 0);
	if (sscanf(line, "%hx", &clist[chars].ucs)) {
	  j = 0;
	  while (isxdigit(line[j])) j++;
	  while (isspace(line[j]) || line[j] == '#') j++;
	  clist[chars].text = malloc(strlen(line + j) + 1);
	  if (clist[chars].text)
	    strcpy(clist[chars].text, line + j);
	  chars++;
	}
      }
      printf("Repertoire of %d characters\n", chars);
    } else if (fonts < MAXFONTS)
      font_name[fonts++] = argv[i];
  }

  if (fonts < 2) {
    fprintf(stderr, usage, version, argv[0]);
    exit(1);
  }

  /* default repertoire */
  if (!chars)
    for (chars = 0; chars < CLISTLEN; chars++) {
      clist[chars].ucs = chars;
      clist[chars].text = "";
    }

  /* connect the X server */
  display = XOpenDisplay(display_name);
  if (!display) {
    fprintf(stderr, "Can't contact X server!\n");
    exit(1);
  }
  screen = DefaultScreen(display);

  /* get all fonts */
  for (i = 0; i < fonts; i++) {
    printf("Loading %s ...\n", font_name[i]);
    font_info[i] = XLoadQueryFont(display, font_name[i]);
    if (!font_info[i]) {
      printf("%s: no such font\n", font_name[i]);
      exit(1);
    }
  }

  fontnamelines = fonts;
  if (fontnamelines > 4)
    fontnamelines = 4;

  /* create a window */
  win = XCreateSimpleWindow(display, RootWindow(display, screen),
                            0, 0, 20 * fonts + 300, 400, 2,
                            BlackPixel(display, screen),
                            WhitePixel(display, screen));

  /* create bitmap for the icon */
  icon_pixmap = XCreateBitmapFromData(display, win, (char *) testglyph_bitmap_bits,
                                      testglyph_bitmap_width,
                                      testglyph_bitmap_height);

  /* define the allowed window size for window manager */
  sizehints.flags = PSize | PMinSize;
  sizehints.width = 20 * fonts + 300; sizehints.height = 400;
  sizehints.min_width = 20 * fonts + 10; sizehints.min_height = 100;

  /* tell the window manager what this window is about */
  XSetStandardProperties(display, win, "glyphtest", "glyphtest", icon_pixmap,
                         argv, argc, &sizehints);
  
  /* select the events we are interested in */
  XSelectInput(display, win, ExposureMask | KeyPressMask);

  /* get a colormap */
  colormap = DefaultColormap(display, screen);

  /* setup graphics contexts */

  for (i = 0; i < fonts; i++) {
    gc_text[i] = XCreateGC(display, win, 0, NULL);
    XSetFont(display, gc_text[i], font_info[i]->fid);
    XSetForeground(display, gc_text[i], BlackPixel(display, screen));
    XSetBackground(display, gc_text[i], WhitePixel(display, screen));
    if (font_info[i]->ascent + font_info[i]->descent > lineskip)
      lineskip = font_info[i]->ascent + font_info[i]->descent;
  }

  gc_text_bg = XCreateGC(display, win, 0, NULL);
  XSetForeground(display, gc_text_bg, WhitePixel(display, screen));
  XSetLineAttributes(display, gc_text[0],
		     1, LineSolid, CapNotLast, JoinMiter);

  /* now send all this to the server */
  XMapWindow(display, win);

  /* main event loop */
  while (1) {
    XNextEvent(display, &event);
    switch (event.type) {
    case Expose:
      redraw();
      break;
    case KeyPress:
      XLookupString((XKeyEvent *)&event, key, KEYLEN, &keysym, &composestatus);
      switch (keysym) {
      case XK_Home:
	firstchar = 0;
	redraw();
	break;
      case XK_Up:
	if (firstchar > 0) {
	  firstchar--;
	  redraw();
	} else
	  XBell(display, 0);
	break;
      case XK_Down:
	if (firstchar < chars-1) {
	  firstchar++;
	  redraw();
	} else
	  XBell(display, 0);
	break;
      case XK_Next:
      case XK_space:
	if (firstchar < chars-1) {
	  firstchar += lines;
	  if (firstchar >= chars)
	    firstchar = chars-1;
	  redraw();
	} else
	  XBell(display, 0);
	break;
      case XK_Prior:
      case XK_BackSpace:
      case XK_Delete:
	if (firstchar > 0) {
	  firstchar -= lines;
	  if (firstchar < 0)
	    firstchar = 0;
	  redraw();
	} else
	  XBell(display, 0);
	break;
      case XK_Right:
	if (firstchar < chars-1) {
	  if (event.xkey.state && ShiftMask)
	      firstchar = (firstchar + 0x1000) & 0x1f000;
	    else
	      firstchar = (firstchar + 0x100) & 0x1ff00;
	  if (firstchar >= chars)
	    firstchar = chars-1;
	  redraw();
	} else
	  XBell(display, 0);
	break;
      case XK_Left:
	if (firstchar > 0) {
	  if (event.xkey.state && ShiftMask)
	      firstchar = (firstchar - 1) & 0x1f000;
	    else
	      firstchar = (firstchar - 1) & 0x1ff00;
	  if (firstchar < 0)
	    firstchar = 0;
	  redraw();
	} else
	  XBell(display, 0);
	break;
      case XK_h:
      case XK_H:
	helper = !helper;
	redraw();
	break;
      case XK_Escape:
      case XK_x:
      case XK_X:
      case XK_q:
      case XK_Q:
        /* say good bye to the server */
        XCloseDisplay(display);
        return 0;
      case XK_Shift_L:
      case XK_Shift_R:
      case XK_Control_L:
      case XK_Control_R:
      case XK_Caps_Lock:
      case XK_Shift_Lock:
        break;
      default:
        XBell(display, 0);
      }
    }
  }

}
